<?php

namespace App\Exports;

use App\Models\User;
use Illuminate\Support\Collection;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Maatwebsite\Excel\Concerns\WithStyles;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;
use PhpOffice\PhpSpreadsheet\Style\Font;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use PhpOffice\PhpSpreadsheet\Style\Fill;

class ApplicantsExport implements FromCollection, WithHeadings, ShouldAutoSize, WithStyles
{
  protected $filters = [];

  public function __construct(array $filters = [])
  {
    $this->filters = $filters;
  }

  public function collection(): Collection
  {
    $query = User::where('role', 'user')
      ->with([
        'applicant' => function ($q) {
          $q->with([
            'governorate',
            'nationalities.country',
            'languages.language',
            'degrees',
            'documents'
          ]);
        },
        'categories',
        'evaluations',
        'examAttempts',
        'evaluationAggregate'
      ])
      ->withCount(['evaluations', 'examAttempts']);

    // Apply filters
    if (isset($this->filters['status']) && $this->filters['status']) {
      $query->where('status', $this->filters['status']);
    }

    if (isset($this->filters['category_id']) && $this->filters['category_id']) {
      $query->whereHas('categories', function ($q) {
        $q->where('category_id', $this->filters['category_id']);
      });
    }

    if (isset($this->filters['search']) && $this->filters['search']) {
      $search = $this->filters['search'];
      $query->where(function ($q) use ($search) {
        $q->where('full_name', 'like', "%{$search}%")
          ->orWhere('email', 'like', "%{$search}%")
          ->orWhereHas('applicant', function ($qa) use ($search) {
            $qa->where('mobile', 'like', "%{$search}%")
              ->orWhere('national_id', 'like', "%{$search}%");
          });
      });
    }

    $users = $query->get();

    return $users->map(function ($user) {
      $applicant = $user->applicant;

      // Get nationalities
      $nationalities = $applicant?->nationalities?->map(fn($n) => $n->country?->name_ar)->join(', ') ?? '';

      // Get languages with proficiency
      $languages = $applicant?->languages?->map(function ($l) {
        $proficiency = $l->proficiency ? " ({$l->proficiency})" : '';
        return $l->language?->name_ar . $proficiency;
      })->join(', ') ?? '';

      // Get degrees
      $degrees = $applicant?->degrees?->map(function ($d) {
        return $d->degree_title . ' - ' . $d->institution . ' (' . $d->year . ')';
      })->join('; ') ?? '';

      // Get evaluation score if exists
      $evaluationScore = $user->evaluationAggregate?->total_score ?? 'لم يتم التقييم';

      return [
        // معلومات المستخدم الأساسية
        'رقم المستخدم' => $user->id,
        'كود المتقدم' => $applicant?->applicant_code,
        'الاسم الكامل (إنجليزي)' => $user->full_name,
        'الاسم الكامل (عربي)' => $applicant?->full_name_ar,
        'البريد الإلكتروني' => $user->email,
        'الحالة' => $this->getStatusLabel($user->status),
        'تاريخ التقديم' => $user->created_at?->format('Y-m-d H:i'),
        'آخر تسجيل دخول' => $user->last_login_at?->format('Y-m-d H:i'),

        // معلومات التعريف
        'الرقم الوطني' => $applicant?->national_id,
        'رقم الجواز' => $applicant?->passport_number,

        // معلومات شخصية
        'تاريخ الميلاد' => $applicant?->dob,
        'الجنس' => $this->getGenderLabel($applicant?->gender),
        'الحالة العسكرية' => $applicant?->military_status,
        'مكان الميلاد' => $applicant?->birth_place,
        'الطول (سم)' => $applicant?->height,
        'الوزن (كجم)' => $applicant?->weight,

        // معلومات الاتصال
        'رقم الهاتف' => $applicant?->mobile,
        'رقم الواتس' => $applicant?->whatsapp,
        'العنوان' => $applicant?->address,
        'الحي/المنطقة' => $applicant?->district,
        'المحافظة' => $applicant?->governorate?->name_ar,

        // معلومات تعليمية
        'الجامعة' => $applicant?->university,
        'الكلية' => $applicant?->college,
        'التخصص' => $applicant?->major,
        'الدرجة العلمية' => $applicant?->degree,
        'هل يدرس حالياً' => $applicant?->is_studying ? 'نعم' : 'لا',
        'سنة الدراسة الحالية' => $applicant?->study_year,
        'سنة التخرج' => $applicant?->grad_year,
        'الدرجات العلمية الإضافية' => $degrees,
        'اللغات' => $languages,
        'الجنسيات الأخرى' => $nationalities,

        // معلومات العمل والخبرة
        'الوظيفة الحالية' => $applicant?->current_job,
        'جهة العمل' => $applicant?->employer,
        'خبرة القطاع العام (سنوات)' => $applicant?->experience_public,
        'خبرة الوزارة (سنوات)' => $applicant?->experience_ministry,
        'خبرة جهات أخرى (سنوات)' => $applicant?->experience_other_orgs,

        // معلومات شخصية إضافية
        'الدول المزارة' => $applicant?->visited_countries,
        'عضويات المراكز' => $applicant?->center_memberships,
        'عضويات الأندية' => $applicant?->club_memberships,
        'الرياضات والألعاب' => $applicant?->sports,
        'المهارات والهوايات' => $applicant?->skills_hobbies,
        'الدافع/التحفيز' => $applicant?->motivation,
        'نقاط القوة' => $applicant?->strengths,
        'نقاط الضعف' => $applicant?->weaknesses,

        // وسائل التواصل الاجتماعي
        'فيسبوك' => $applicant?->facebook,
        'إنستجرام' => $applicant?->instagram,
        'لينكدإن' => $applicant?->linkedin,

        // معلومات الاختبار والتقييم
        'عدد محاولات الاختبار' => $user->exam_attempts_count,
        'عدد التقييمات المكتملة' => $user->evaluations_count,
        'درجة التقييم النهائية' => $evaluationScore,

        // معلومات التصنيفات
        'التصنيفات' => $user->categories->pluck('name')->join(', '),
        'الحالة النشطة' => $user->is_active ? 'نعم' : 'لا',
      ];
    });
  }

  public function headings(): array
  {
    return [
      // معلومات المستخدم الأساسية
      'رقم المستخدم',
      'كود المتقدم',
      'الاسم الكامل (إنجليزي)',
      'الاسم الكامل (عربي)',
      'البريد الإلكتروني',
      'الحالة',
      'تاريخ التقديم',
      'آخر تسجيل دخول',

      // معلومات التعريف
      'الرقم الوطني',
      'رقم الجواز',

      // معلومات شخصية
      'تاريخ الميلاد',
      'الجنس',
      'الحالة العسكرية',
      'مكان الميلاد',
      'الطول (سم)',
      'الوزن (كجم)',

      // معلومات الاتصال
      'رقم الهاتف',
      'رقم الواتس',
      'العنوان',
      'الحي/المنطقة',
      'المحافظة',

      // معلومات تعليمية
      'الجامعة',
      'الكلية',
      'التخصص',
      'الدرجة العلمية',
      'هل يدرس حالياً',
      'سنة الدراسة الحالية',
      'سنة التخرج',
      'الدرجات العلمية الإضافية',
      'اللغات',
      'الجنسيات الأخرى',

      // معلومات العمل والخبرة
      'الوظيفة الحالية',
      'جهة العمل',
      'خبرة القطاع العام (سنوات)',
      'خبرة الوزارة (سنوات)',
      'خبرة جهات أخرى (سنوات)',

      // معلومات شخصية إضافية
      'الدول المزارة',
      'عضويات المراكز',
      'عضويات الأندية',
      'الرياضات والألعاب',
      'المهارات والهوايات',
      'الدافع/التحفيز',
      'نقاط القوة',
      'نقاط الضعف',

      // وسائل التواصل الاجتماعي
      'فيسبوك',
      'إنستجرام',
      'لينكدإن',

      // معلومات الاختبار والتقييم
      'عدد محاولات الاختبار',
      'عدد التقييمات المكتملة',
      'درجة التقييم النهائية',

      // معلومات التصنيفات
      'التصنيفات',
      'الحالة النشطة',
    ];
  }

  public function styles(Worksheet $sheet): array
  {
    return [
      1 => [
        'font' => [
          'bold' => true,
          'color' => ['rgb' => 'FFFFFF'],
          'size' => 11,
        ],
        'fill' => [
          'fillType' => Fill::FILL_SOLID,
          'startColor' => ['rgb' => '1F2937'],
        ],
        'alignment' => [
          'horizontal' => Alignment::HORIZONTAL_CENTER,
          'vertical' => Alignment::VERTICAL_CENTER,
          'wrapText' => true,
        ],
      ],
    ];
  }

  private function getStatusLabel($status): string
  {
    $labels = [
      'pending' => 'قيد الانتظار',
      'exam_completed' => 'أكمل الاختبار',
      'under_evaluation' => 'تحت التقييم',
      'evaluated' => 'تم التقييم',
      'accepted' => 'مقبول',
      'rejected' => 'مرفوض',
    ];
    return $labels[$status] ?? $status;
  }

  private function getGenderLabel($gender): string
  {
    $labels = [
      'male' => 'ذكر',
      'female' => 'أنثى',
      'other' => 'آخر',
    ];
    return $labels[$gender] ?? ($gender ?? '');
  }
}
