<?php

namespace App\Http\Controllers;

use App\Models\Category;
use App\Models\User;
use App\Services\CategoryService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Auth;

class CategoryController extends Controller
{
  public function __construct(
    protected CategoryService $categoryService,
    protected \App\Services\AuditService $auditService,
  ) {}

  /**
   * Get all categories
   */
  public function index(): JsonResponse
  {
    $categories = Category::withCount('users')->get();

    return response()->json([
      'success' => true,
      'data' => $categories,
    ]);
  }

  /**
   * Create a new category
   */
  public function store(Request $request): JsonResponse
  {
    $request->validate([
      'name' => 'required|string|max:255',
      'description' => 'nullable|string',
    ]);

    $admin = Auth::user();
    $category = $this->categoryService->createCategory($request->all(), $admin);

    return response()->json([
      'success' => true,
      'message' => 'Category created successfully',
      'data' => $category,
    ], 201);
  }

  /**
   * Update an existing category
   */
  public function update(Request $request, Category $category): JsonResponse
  {
    $request->validate([
      'name' => 'required|string|max:255',
      'description' => 'nullable|string',
    ]);

    $admin = Auth::user();
    $category->update($request->only(['name', 'description']));

    $this->auditService->log(
      causer: $admin,
      action: 'category.updated',
      subject: $category,
      meta: ['name' => $category->name]
    );

    return response()->json([
      'success' => true,
      'message' => 'Category updated successfully',
      'data' => $category,
    ]);
  }

  /**
   * Delete a category
   */
  public function destroy(Category $category): JsonResponse
  {
    // Check if category has users
    if ($category->users()->count() > 0) {
      return response()->json([
        'success' => false,
        'message' => 'Cannot delete category with assigned users. Please remove all users first.',
      ], 422);
    }

    $admin = Auth::user();
    $this->auditService->log(
      causer: $admin,
      action: 'category.deleted',
      subject: $category,
      meta: ['name' => $category->name]
    );

    $category->delete();

    return response()->json([
      'success' => true,
      'message' => 'Category deleted successfully',
    ]);
  }

  /**
   * Get users in a category
   */
  public function users(Category $category): JsonResponse
  {
    $users = $this->categoryService->getCategoryUsers($category);

    return response()->json([
      'success' => true,
      'data' => $users,
    ]);
  }

  /**
   * Assign a user to a category
   */
  public function assignUser(Request $request, Category $category): JsonResponse
  {
    $request->validate([
      'user_id' => 'required|exists:users,id',
    ]);

    $user = User::findOrFail($request->user_id);
    $admin = Auth::user();

    $this->categoryService->assignUserToCategory($user, $category, $admin);

    return response()->json([
      'success' => true,
      'message' => 'User assigned to category successfully',
    ]);
  }

  /**
   * Remove a user from a category
   */
  public function removeUser(Request $request, Category $category): JsonResponse
  {
    $request->validate([
      'user_id' => 'required|exists:users,id',
    ]);

    $user = User::findOrFail($request->user_id);
    $admin = Auth::user();

    $this->categoryService->removeUserFromCategory($user, $category, $admin);

    return response()->json([
      'success' => true,
      'message' => 'User removed from category successfully',
    ]);
  }

  /**
   * Bulk assign users to a category
   */
  public function bulkAssignUsers(Request $request, Category $category): JsonResponse
  {
    $request->validate([
      'user_ids' => 'required|array',
      'user_ids.*' => 'exists:users,id',
    ]);

    $admin = Auth::user();
    $count = $this->categoryService->bulkAssignUsersToCategory($request->user_ids, $category, $admin);

    return response()->json([
      'success' => true,
      'message' => "{$count} users assigned to category successfully",
      'data' => ['count' => $count],
    ]);
  }

  /**
   * Bulk remove users from a category
   */
  public function bulkRemoveUsers(Request $request, Category $category): JsonResponse
  {
    $request->validate([
      'user_ids' => 'required|array',
      'user_ids.*' => 'exists:users,id',
    ]);

    $admin = Auth::user();
    $count = $this->categoryService->bulkRemoveUsersFromCategory($request->user_ids, $category, $admin);

    return response()->json([
      'success' => true,
      'message' => "{$count} users removed from category successfully",
      'data' => ['count' => $count],
    ]);
  }
}
