<?php

namespace App\Http\Controllers;

use App\Models\Country;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Auth;

class CountryController extends Controller
{
  public function __construct(
    protected \App\Services\AuditService $auditService,
  ) {}

  /**
   * Get all countries
   */
  public function index(): JsonResponse
  {
    $countries = Country::orderBy('name_en')->get();

    return response()->json([
      'success' => true,
      'data' => $countries,
    ]);
  }

  /**
   * Create a new country
   */
  public function store(Request $request): JsonResponse
  {
    $request->validate([
      'name_en' => 'required|string|max:255|unique:countries,name_en',
      'name_ar' => 'required|string|max:255|unique:countries,name_ar',
      'iso_code' => 'nullable|string|max:5|unique:countries,iso_code',
    ]);

    $country = Country::create($request->only(['name_en', 'name_ar', 'iso_code']));

    $admin = Auth::user();
    $this->auditService->log(
      causer: $admin,
      action: 'country.created',
      subject: $country,
      meta: ['name_en' => $country->name_en, 'name_ar' => $country->name_ar]
    );

    return response()->json([
      'success' => true,
      'message' => 'Country created successfully',
      'data' => $country,
    ], 201);
  }

  /**
   * Get a specific country
   */
  public function show(Country $country): JsonResponse
  {
    return response()->json([
      'success' => true,
      'data' => $country,
    ]);
  }

  /**
   * Update a country
   */
  public function update(Request $request, Country $country): JsonResponse
  {
    $request->validate([
      'name_en' => 'required|string|max:255|unique:countries,name_en,' . $country->id,
      'name_ar' => 'required|string|max:255|unique:countries,name_ar,' . $country->id,
      'iso_code' => 'nullable|string|max:5|unique:countries,iso_code,' . $country->id,
    ]);

    $oldData = $country->only(['name_en', 'name_ar', 'iso_code']);
    $country->update($request->only(['name_en', 'name_ar', 'iso_code']));

    $admin = Auth::user();
    $this->auditService->log(
      causer: $admin,
      action: 'country.updated',
      subject: $country,
      meta: ['old' => $oldData, 'new' => $country->only(['name_en', 'name_ar', 'iso_code'])]
    );

    return response()->json([
      'success' => true,
      'message' => 'Country updated successfully',
      'data' => $country,
    ]);
  }

  /**
   * Delete a country
   */
  public function destroy(Country $country): JsonResponse
  {
    $admin = Auth::user();
    $countryName = $country->name_en;

    $this->auditService->log(
      causer: $admin,
      action: 'country.deleted',
      subject: $country,
      meta: ['name_en' => $countryName, 'name_ar' => $country->name_ar]
    );

    $country->delete();

    return response()->json([
      'success' => true,
      'message' => 'Country deleted successfully',
    ]);
  }
}
