<?php

namespace App\Http\Controllers;

use App\Models\Governorate;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Auth;

class GovernorateController extends Controller
{
  public function __construct(
    protected \App\Services\AuditService $auditService,
  ) {}

  /**
   * Get all governorates
   */
  public function index(): JsonResponse
  {
    $governorates = Governorate::orderBy('name_en')->get();

    return response()->json([
      'success' => true,
      'data' => $governorates,
    ]);
  }

  /**
   * Create a new governorate
   */
  public function store(Request $request): JsonResponse
  {
    $request->validate([
      'name_en' => 'required|string|max:255|unique:governorates,name_en',
      'name_ar' => 'required|string|max:255|unique:governorates,name_ar',
      'code' => 'nullable|string|max:255|unique:governorates,code',
    ]);

    $governorate = Governorate::create($request->only(['name_en', 'name_ar', 'code']));

    $admin = Auth::user();
    $this->auditService->log(
      causer: $admin,
      action: 'governorate.created',
      subject: $governorate,
      meta: ['name_en' => $governorate->name_en, 'name_ar' => $governorate->name_ar]
    );

    return response()->json([
      'success' => true,
      'message' => 'Governorate created successfully',
      'data' => $governorate,
    ], 201);
  }

  /**
   * Get a specific governorate
   */
  public function show(Governorate $governorate): JsonResponse
  {
    return response()->json([
      'success' => true,
      'data' => $governorate,
    ]);
  }

  /**
   * Update a governorate
   */
  public function update(Request $request, Governorate $governorate): JsonResponse
  {
    $request->validate([
      'name_en' => 'required|string|max:255|unique:governorates,name_en,' . $governorate->id,
      'name_ar' => 'required|string|max:255|unique:governorates,name_ar,' . $governorate->id,
      'code' => 'nullable|string|max:255|unique:governorates,code,' . $governorate->id,
    ]);

    $oldData = $governorate->only(['name_en', 'name_ar', 'code']);
    $governorate->update($request->only(['name_en', 'name_ar', 'code']));

    $admin = Auth::user();
    $this->auditService->log(
      causer: $admin,
      action: 'governorate.updated',
      subject: $governorate,
      meta: ['old' => $oldData, 'new' => $governorate->only(['name_en', 'name_ar', 'code'])]
    );

    return response()->json([
      'success' => true,
      'message' => 'Governorate updated successfully',
      'data' => $governorate,
    ]);
  }

  /**
   * Delete a governorate
   */
  public function destroy(Governorate $governorate): JsonResponse
  {
    $admin = Auth::user();
    $governorateName = $governorate->name_en;

    $this->auditService->log(
      causer: $admin,
      action: 'governorate.deleted',
      subject: $governorate,
      meta: ['name_en' => $governorateName, 'name_ar' => $governorate->name_ar]
    );

    $governorate->delete();

    return response()->json([
      'success' => true,
      'message' => 'Governorate deleted successfully',
    ]);
  }
}
