<?php

namespace App\Http\Controllers;

use App\Models\Language;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Auth;

class LanguageController extends Controller
{
  public function __construct(
    protected \App\Services\AuditService $auditService,
  ) {}

  /**
   * Get all languages
   */
  public function index(): JsonResponse
  {
    $languages = Language::orderBy('name_en')->get();

    return response()->json([
      'success' => true,
      'data' => $languages,
    ]);
  }

  /**
   * Create a new language
   */
  public function store(Request $request): JsonResponse
  {
    $request->validate([
      'name_en' => 'required|string|max:255|unique:languages,name_en',
      'name_ar' => 'required|string|max:255|unique:languages,name_ar',
      'iso_code' => 'nullable|string|max:10|unique:languages,iso_code',
    ]);

    $language = Language::create($request->only(['name_en', 'name_ar', 'iso_code']));

    $admin = Auth::user();
    $this->auditService->log(
      causer: $admin,
      action: 'language.created',
      subject: $language,
      meta: ['name_en' => $language->name_en, 'name_ar' => $language->name_ar]
    );

    return response()->json([
      'success' => true,
      'message' => 'Language created successfully',
      'data' => $language,
    ], 201);
  }

  /**
   * Get a specific language
   */
  public function show(Language $language): JsonResponse
  {
    return response()->json([
      'success' => true,
      'data' => $language,
    ]);
  }

  /**
   * Update a language
   */
  public function update(Request $request, Language $language): JsonResponse
  {
    $request->validate([
      'name_en' => 'required|string|max:255|unique:languages,name_en,' . $language->id,
      'name_ar' => 'required|string|max:255|unique:languages,name_ar,' . $language->id,
      'iso_code' => 'nullable|string|max:10|unique:languages,iso_code,' . $language->id,
    ]);

    $oldData = $language->only(['name_en', 'name_ar', 'iso_code']);
    $language->update($request->only(['name_en', 'name_ar', 'iso_code']));

    $admin = Auth::user();
    $this->auditService->log(
      causer: $admin,
      action: 'language.updated',
      subject: $language,
      meta: ['old' => $oldData, 'new' => $language->only(['name_en', 'name_ar', 'iso_code'])]
    );

    return response()->json([
      'success' => true,
      'message' => 'Language updated successfully',
      'data' => $language,
    ]);
  }

  /**
   * Delete a language
   */
  public function destroy(Language $language): JsonResponse
  {
    $admin = Auth::user();
    $languageName = $language->name_en;

    $this->auditService->log(
      causer: $admin,
      action: 'language.deleted',
      subject: $language,
      meta: ['name_en' => $languageName, 'name_ar' => $language->name_ar]
    );

    $language->delete();

    return response()->json([
      'success' => true,
      'message' => 'Language deleted successfully',
    ]);
  }
}
