<?php

namespace App\Http\Controllers;

use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;

class SettingController extends Controller
{
    public function __construct(
        protected \App\Services\AuditService $auditService,
    ) {}
    /**
     * Get all settings
     */
    public function index()
    {
        try {
            $settings = Setting::getAll();
            return response()->json([
                'status' => true,
                'message' => 'Settings retrieved successfully',
                'data' => $settings,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'Failed to fetch settings',
            ], 500);
        }
    }

    /**
     * Update settings
     */
    public function update(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'settings' => 'required|array',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors(),
            ], 422);
        }

        try {
            $admin = Auth::user();
            $updatedSettings = [];

            foreach ($request->settings as $key => $value) {
                $type = $this->getSettingType($key);
                Setting::set($key, $value, $type);
                $updatedSettings[$key] = $value;
            }

            $this->auditService->log(
                causer: $admin,
                action: 'setting.updated',
                subject: null,
                meta: [
                    'updated_settings' => $updatedSettings,
                    'count' => count($updatedSettings),
                ]
            );

            return response()->json([
                'status' => true,
                'message' => 'Settings updated successfully',
                'data' => Setting::getAll(),
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'Failed to update settings',
            ], 500);
        }
    }

    /**
     * Get a specific setting
     */
    public function show(string $key)
    {
        try {
            $value = Setting::get($key);

            if ($value === null) {
                return response()->json([
                    'status' => false,
                    'message' => 'Setting not found',
                ], 404);
            }

            return response()->json([
                'status' => true,
                'data' => ['key' => $key, 'value' => $value],
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'Failed to fetch setting',
            ], 500);
        }
    }

    /**
     * Determine setting type based on key
     */
    private function getSettingType(string $key): string
    {
        $booleanSettings = [
            'registration_open',
            'maintenance_mode',
            'email_notifications',
            'sms_notifications',
            'auto_publish_results',
        ];

        $integerSettings = [
            'exam_duration',
            'passing_score',
            'max_login_attempts',
        ];

        if (in_array($key, $booleanSettings)) {
            return 'boolean';
        }

        if (in_array($key, $integerSettings)) {
            return 'integer';
        }

        return 'string';
    }
}
