<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Services\UserReviewService;
use App\Exports\ApplicantsExport;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Auth;

class UserReviewController extends Controller
{
  public function __construct(
    protected UserReviewService $reviewService,
    protected \App\Services\AuditService $auditService,
  ) {}

  /**
   * Get all users with filters
   */
  public function index(Request $request): JsonResponse
  {
    $query = User::where('role', 'user')
      ->with('categories')
      ->withCount(['evaluations', 'examAttempts']);

    // Filter by status
    if ($request->has('status')) {
      $query->where('status', $request->status);
    }

    // Filter by category
    if ($request->has('category_id')) {
      $query->whereHas('categories', function ($q) use ($request) {
        $q->where('category_id', $request->category_id);
      });
    }

    // Search by name or email
    if ($request->has('search')) {
      $search = $request->search;
      $query->where(function ($q) use ($search) {
        $q->where('full_name', 'like', "%{$search}%")
          ->orWhere('email', 'like', "%{$search}%")
          ->orWhereHas('applicant', function ($qa) use ($search) {
            $qa->where('mobile', 'like', "%{$search}%")
              ->orWhere('national_id', 'like', "%{$search}%");
          });
      });
    }

    $users = $query->paginate($request->per_page ?? 15);

    return response()->json([
      'success' => true,
      'data' => $users,
    ]);
  }

  /**
   * Get comprehensive review data for a user
   */
  public function show(User $user): JsonResponse
  {
    $reviewData = $this->reviewService->getUserReviewData($user);

    return response()->json([
      'success' => true,
      'data' => $reviewData,
    ]);
  }

  /**
   * Accept a user
   */
  public function accept(Request $request, User $user): JsonResponse
  {
    $request->validate([
      'notes' => 'nullable|string',
    ]);

    try {
      $reviewer = Auth::user();
      $this->reviewService->acceptUser($user, $reviewer, $request->notes);

      return response()->json([
        'success' => true,
        'message' => 'User accepted successfully',
      ]);
    } catch (\Exception $e) {
      return response()->json([
        'success' => false,
        'message' => $e->getMessage(),
      ], 400);
    }
  }

  /**
   * Reject a user
   */
  public function reject(Request $request, User $user): JsonResponse
  {
    $request->validate([
      'reason' => 'nullable|string',
    ]);

    try {
      $reviewer = Auth::user();
      $this->reviewService->rejectUser($user, $reviewer, $request->reason);

      return response()->json([
        'success' => true,
        'message' => 'User rejected successfully',
      ]);
    } catch (\Exception $e) {
      return response()->json([
        'success' => false,
        'message' => $e->getMessage(),
      ], 400);
    }
  }

  /**
   * Get user statistics
   */
  public function statistics(Request $request): JsonResponse
  {
    $categoryId = $request->category_id;
    $stats = $this->reviewService->getUserStatistics($categoryId);

    return response()->json([
      'success' => true,
      'data' => $stats,
    ]);
  }

  /**
   * Get user's applicant registration data
   */
  public function applicantData(User $user): JsonResponse
  {
    $applicant = $user->applicant()->with(['nationalities', 'languages', 'degrees', 'documents'])->first();

    if (!$applicant) {
      return response()->json([
        'success' => false,
        'message' => 'No applicant data found for this user',
      ], 404);
    }

    return response()->json([
      'success' => true,
      'data' => $applicant,
    ]);
  }

  /**
   * Get user's exam attempt
   */
  public function examAttempt(User $user): JsonResponse
  {
    $attempt = $user->examAttempts()->with('exam')->latest()->first();

    if (!$attempt) {
      return response()->json([
        'success' => false,
        'message' => 'No exam attempt found for this user',
      ], 404);
    }

    return response()->json([
      'success' => true,
      'data' => $attempt,
    ]);
  }

  /**
   * Get user's detailed exam results
   */
  public function examResults(User $user): JsonResponse
  {
    $attempt = $user->examAttempts()->with('exam')->latest()->first();

    if (!$attempt || !$attempt->completed_at) {
      return response()->json([
        'success' => false,
        'message' => 'No completed exam attempt found for this user',
      ], 404);
    }

    $exam = $attempt->exam;
    $passingScore = \App\Models\Setting::get('passing_score', 60);
    $userAnswers = $attempt->answers ?? [];

    // Build detailed results with questions and answers
    $questions = $exam->form_schema['questions'] ?? [];
    $detailedQuestions = array_map(function ($question) use ($userAnswers) {
      $questionId = $question['id'];
      $userAnswer = $userAnswers[$questionId] ?? null;
      $correctAnswer = $question['correct_answer'] ?? null;

      // Determine if answer is correct
      $isCorrect = $userAnswer && $correctAnswer && $userAnswer === $correctAnswer;

      return [
        'id' => $question['id'],
        'text' => $question['text'] ?? $question['question'] ?? '',
        'type' => $question['type'] ?? 'multiple_choice',
        'user_answer' => $userAnswer,
        'correct_answer' => $correctAnswer,
        'is_correct' => $isCorrect,
        'options' => $question['options'] ?? [],
      ];
    }, $questions);

    return response()->json([
      'success' => true,
      'data' => [
        'attempt' => [
          'id' => $attempt->id,
          'exam_id' => $attempt->exam_id,
          'exam_title' => $exam->title,
          'score' => $attempt->score,
          'passing_score' => $passingScore,
          'started_at' => $attempt->started_at,
          'completed_at' => $attempt->completed_at,
          'auto_submitted' => $attempt->auto_submitted,
        ],
        'exam' => [
          'id' => $exam->id,
          'title' => $exam->title,
          'passing_score' => $passingScore,
        ],
        'results' => [
          'score' => (float) $attempt->score,
          'passing_score' => $passingScore,
          'passed' => ((float) $attempt->score) >= $passingScore,
          'total_questions' => count($questions),
          'correct_answers' => count(array_filter($detailedQuestions, fn($q) => $q['is_correct'])),
          'wrong_answers' => count(array_filter($detailedQuestions, fn($q) => !$q['is_correct'])),
        ],
        'questions' => $detailedQuestions,
      ],
    ]);
  }

  /**
   * Assign categories to a user
   */
  public function assignCategories(Request $request, User $user): JsonResponse
  {
    $request->validate([
      'category_ids' => 'required|array',
      'category_ids.*' => 'exists:categories,id',
    ]);

    $admin = Auth::user();
    $count = 0;
    $assignedIds = [];

    foreach ($request->category_ids as $categoryId) {
      if (!$user->categories->contains($categoryId)) {
        $user->categories()->attach($categoryId);
        $count++;
        $assignedIds[] = $categoryId;
      }
    }

    if ($count > 0) {
      $this->auditService->log(
        causer: $admin,
        action: 'user.categories_assigned',
        subject: $user,
        meta: [
          'user_id' => $user->id,
          'category_ids' => $assignedIds,
          'count' => $count,
        ]
      );
    }

    return response()->json([
      'success' => true,
      'message' => "{$count} categories assigned to user successfully",
      'data' => ['count' => $count],
    ]);
  }

  /**
   * Remove categories from a user
   */
  public function removeCategories(Request $request, User $user): JsonResponse
  {
    $request->validate([
      'category_ids' => 'required|array',
      'category_ids.*' => 'exists:categories,id',
    ]);

    $admin = Auth::user();
    $count = 0;
    $removedIds = [];

    foreach ($request->category_ids as $categoryId) {
      if ($user->categories->contains($categoryId)) {
        $user->categories()->detach($categoryId);
        $count++;
        $removedIds[] = $categoryId;
      }
    }

    if ($count > 0) {
      $this->auditService->log(
        causer: $admin,
        action: 'user.categories_removed',
        subject: $user,
        meta: [
          'user_id' => $user->id,
          'category_ids' => $removedIds,
          'count' => $count,
        ]
      );
    }

    return response()->json([
      'success' => true,
      'message' => "{$count} categories removed from user successfully",
      'data' => ['count' => $count],
    ]);
  }

  /**
   * Export applicants data to Excel
   */
  public function exportToExcel(Request $request)
  {
    try {
      $filters = [
        'status' => $request->query('status'),
        'category_id' => $request->query('category_id'),
        'search' => $request->query('search'),
      ];

      $fileName = 'applicants_' . now()->format('Y_m_d_H_i_s') . '.xlsx';

      return Excel::download(
        new ApplicantsExport($filters),
        $fileName
      );
    } catch (\Exception $e) {
      return response()->json([
        'success' => false,
        'message' => 'Failed to export data: ' . $e->getMessage(),
      ], 500);
    }
  }
}
