<?php

namespace App\Services;

use App\Models\Evaluation;
use App\Models\EvaluationAggregate;
use App\Models\EvaluationSheet;
use App\Models\User;
use App\Jobs\AggregateEvaluationScores;

class EvaluationService
{
    public function __construct(
        protected AuditService $auditService,
    ) {}

    /**
     * Create an evaluation sheet by a judge
     */
    public function createEvaluationSheet(array $data, User $judge): EvaluationSheet
    {
        $sheet = EvaluationSheet::create([
            'judge_id' => $judge->id,
            'title' => $data['title'],
            'criteria' => $data['criteria'],
            'is_active' => $data['is_active'] ?? true,
        ]);

        $this->auditService->log(
            causer: $judge,
            action: 'evaluation_sheet.created',
            subject: $sheet,
            meta: ['title' => $sheet->title]
        );

        return $sheet;
    }

    /**
     * Submit an evaluation for a user
     */
    public function submitEvaluation(array $data, User $judge, User $evaluatedUser): Evaluation
    {
        // Check if user already has 5 evaluations
        $evaluationCount = $evaluatedUser->getEvaluationCount();

        if ($evaluationCount >= 5) {
            throw new \Exception('User already has the maximum number of evaluations (5)');
        }

        // Check if this judge already evaluated this user
        $existingEvaluation = Evaluation::where('judge_id', $judge->id)
            ->where('evaluated_user_id', $evaluatedUser->id)
            ->first();

        if ($existingEvaluation) {
            throw new \Exception('You have already evaluated this user');
        }

        $score = $this->calculateEvaluationScore($data['answers']);

        $evaluation = Evaluation::create([
            'evaluation_sheet_id' => $data['evaluation_sheet_id'],
            'judge_id' => $judge->id,
            'evaluated_user_id' => $evaluatedUser->id,
            'answers' => $data['answers'],
            'score' => $score,
            'submitted_at' => now(),
        ]);

        // Update user status
        $evaluatedUser->update(['status' => 'under_evaluation']);

        $this->auditService->log(
            causer: $judge,
            action: 'evaluation.submitted',
            subject: $evaluation,
            meta: [
                'evaluated_user_id' => $evaluatedUser->id,
                'score' => $score,
            ]
        );

        // Check if this was the 5th evaluation
        $newCount = $evaluatedUser->getEvaluationCount();
        if ($newCount === 5) {
            // Dispatch job to aggregate scores
            AggregateEvaluationScores::dispatch($evaluatedUser);
        }

        return $evaluation;
    }

    /**
     * Calculate evaluation score from answers
     */
    private function calculateEvaluationScore(array $answers): float
    {
        if (empty($answers)) {
            return 0;
        }

        return array_sum($answers) / count($answers);
    }

    /**
     * Get all evaluations for a user
     */
    public function getUserEvaluations(User $user): \Illuminate\Database\Eloquent\Collection
    {
        return Evaluation::where('evaluated_user_id', $user->id)
            ->with(['judge', 'evaluationSheet'])
            ->orderBy('submitted_at', 'desc')
            ->get();
    }

    public function aggregateEvaluations(User $user): EvaluationAggregate
    {
        $evaluations = Evaluation::where('evaluated_user_id', $user->id)->get();

        if ($evaluations->count() !== 5) {
            throw new \Exception('User must have exactly 5 evaluations to aggregate');
        }

        $criterionAverages = $this->calculateCriterionAverages($evaluations);
        $overallScore = $this->calculateOverallScore($criterionAverages);

        $aggregate = EvaluationAggregate::updateOrCreate(
            ['evaluated_user_id' => $user->id],
            [
                'total_evaluations' => 5,
                'criterion_averages' => $criterionAverages,
                'overall_score' => $overallScore,
                'completed_at' => now(),
            ]
        );

        // Update user status to evaluated
        $user->update(['status' => 'evaluated']);

        $this->auditService->log(
            causer: $user,
            action: 'evaluation.aggregated',
            subject: $aggregate,
            meta: [
                'user_id' => $user->id,
                'overall_score' => $overallScore,
            ]
        );

        return $aggregate;
    }

    private function calculateCriterionAverages(iterable $evaluations): array
    {
        $criterionScores = [];

        foreach ($evaluations as $evaluation) {
            foreach ($evaluation->answers as $criterion => $score) {
                if (!isset($criterionScores[$criterion])) {
                    $criterionScores[$criterion] = [];
                }
                $criterionScores[$criterion][] = $score;
            }
        }

        $averages = [];
        foreach ($criterionScores as $criterion => $scores) {
            $averages[$criterion] = array_sum($scores) / count($scores);
        }

        return $averages;
    }

    private function calculateOverallScore(array $criterionAverages): float
    {
        if (empty($criterionAverages)) {
            return 0;
        }

        return array_sum($criterionAverages) / count($criterionAverages);
    }
}
