<?php

namespace App\Services;

use App\Models\Exam;
use App\Models\ExamAttempt;
use App\Models\User;
use Illuminate\Support\Facades\DB;

class ExamService
{
  public function __construct(
    protected AuditService $auditService,
  ) {}

  /**
   * Publish an exam - makes it available to all users
   */
  public function publishExam(Exam $exam, User $admin): Exam
  {
    $exam->update([
      'is_published' => true,
      'published_at' => now(),
    ]);

    $this->auditService->log(
      causer: $admin,
      action: 'exam.published',
      subject: $exam,
      meta: ['exam_title' => $exam->title]
    );

    return $exam;
  }

  /**
   * Start an exam attempt for a user
   */
  public function startExamAttempt(Exam $exam, User $user): ExamAttempt
  {
    // Check if exam is published
    if (!$exam->is_published) {
      throw new \Exception('Exam is not published yet');
    }

    // Check if user already has an attempt
    $existingAttempt = ExamAttempt::where('exam_id', $exam->id)
      ->where('user_id', $user->id)
      ->first();

    if ($existingAttempt) {
      throw new \Exception('User has already attempted this exam');
    }

    // Get duration from settings or exam default
    $durationMinutes = \App\Models\Setting::get('exam_duration', 60);

    $attempt = ExamAttempt::create([
      'exam_id' => $exam->id,
      'user_id' => $user->id,
      'answers' => [],
      'started_at' => now(),
    ]);

    $this->auditService->log(
      causer: $user,
      action: 'exam.attempt_started',
      subject: $attempt,
      meta: ['exam_id' => $exam->id]
    );

    // Return attempt with exam data including duration
    $attempt->load('exam');
    $attempt->exam->duration_minutes = $durationMinutes;

    // Remove correct answers from schema before returning
    $examData = $attempt->exam->toArray();
    $examData['form_schema'] = $this->sanitizeExamSchema($examData['form_schema']);
    $attempt->exam->form_schema = $examData['form_schema'];

    return $attempt;
  }

  /**
   * Complete an exam attempt
   */
  public function completeExamAttempt(ExamAttempt $attempt, array $answers, bool $autoSubmit = false): ExamAttempt
  {
    $score = $this->calculateScore($attempt->exam, $answers);

    $attempt->update([
      'answers' => $answers,
      'score' => $score,
      'completed_at' => now(),
      'auto_submitted' => $autoSubmit,
    ]);

    // Update user status
    $attempt->user->update(['status' => 'exam_completed']);

    $this->auditService->log(
      causer: $attempt->user,
      action: $autoSubmit ? 'exam.auto_submitted' : 'exam.attempt_completed',
      subject: $attempt,
      meta: [
        'exam_id' => $attempt->exam_id,
        'score' => $score,
        'auto_submitted' => $autoSubmit,
      ]
    );

    return $attempt;
  }

  /**
   * Calculate exam score based on answers
   */
  private function calculateScore(Exam $exam, array $answers): float
  {
    $schema = $exam->form_schema;
    $totalScore = 0;
    $totalPossible = 0;

    foreach ($schema['questions'] ?? [] as $question) {
      $questionId = $question['id'];
      $points = $question['points'] ?? 1;
      $totalPossible += $points;

      if (isset($answers[$questionId])) {
        $userAnswer = $answers[$questionId];
        $correctAnswer = $question['correct_answer'] ?? null;

        if ($correctAnswer && $userAnswer === $correctAnswer) {
          $totalScore += $points;
        }
      }
    }

    return $totalPossible > 0 ? ($totalScore / $totalPossible) * 100 : 0;
  }

  /**
   * Get exam attempt for a user
   */
  public function getUserExamAttempt(Exam $exam, User $user): ?ExamAttempt
  {
    return ExamAttempt::where('exam_id', $exam->id)
      ->where('user_id', $user->id)
      ->first();
  }

  /**
   * Check if user can take exam
   */
  public function canUserTakeExam(Exam $exam, User $user): bool
  {
    if (!$exam->is_published) {
      return false;
    }

    $existingAttempt = $this->getUserExamAttempt($exam, $user);
    return $existingAttempt === null;
  }

  /**
   * Remove correct answers from exam schema
   */
  private function sanitizeExamSchema(array $schema): array
  {
    if (isset($schema['questions'])) {
      $schema['questions'] = array_map(function ($question) {
        unset($question['correct_answer']);
        return $question;
      }, $schema['questions']);
    }
    return $schema;
  }

  /**
   * Check if exam time has expired
   */
  public function isExamTimeExpired(ExamAttempt $attempt): bool
  {
    if (!$attempt->started_at || $attempt->completed_at) {
      return false;
    }

    $durationMinutes = \App\Models\Setting::get('exam_duration', 60);
    $endTime = $attempt->started_at->addMinutes($durationMinutes);

    return now()->greaterThan($endTime);
  }

  /**
   * Get remaining time in seconds
   */
  public function getRemainingTime(ExamAttempt $attempt): int
  {
    if (!$attempt->started_at || $attempt->completed_at) {
      return 0;
    }

    $durationMinutes = \App\Models\Setting::get('exam_duration', 60);
    $endTime = $attempt->started_at->addMinutes($durationMinutes);
    $remainingSeconds = now()->diffInSeconds($endTime, false);

    return max(0, (int) $remainingSeconds);
  }
}
