<?php

namespace App\Services;

use App\Models\User;
use App\Models\Applicant;
use App\Models\ExamAttempt;
use App\Models\EvaluationAggregate;

class UserReviewService
{
  public function __construct(
    protected AuditService $auditService,
  ) {}

  /**
   * Get comprehensive user profile for review
   */
  public function getUserReviewData(User $user): array
  {
    return [
      'user' => [
        'id' => $user->id,
        'email' => $user->email,
        'full_name' => $user->full_name,
        'status' => $user->status,
        'created_at' => $user->created_at,
      ],
      'applicant_data' => $this->getApplicantData($user),
      'exam_result' => $this->getExamData($user),
      'evaluations' => $this->getEvaluationsData($user),
      'aggregate' => $this->getAggregateData($user),
      'categories' => $user->categories->map(function ($category) {
        return [
          'id' => $category->id,
          'name' => $category->name,
        ];
      })->values(),
    ];
  }

  /**
   * Get applicant registration data for user
   */
  private function getApplicantData(User $user): ?array
  {
    $applicant = $user->applicant;

    if (!$applicant) {
      return null;
    }

    return [
      'id' => $applicant->id,
      'applicant_code' => $applicant->applicant_code,
      'full_name_ar' => $applicant->full_name_ar,
      'full_name_en' => $applicant->full_name_en,
      'national_id' => $applicant->national_id,
      'email' => $applicant->email,
      'mobile' => $applicant->mobile,
      'whatsapp' => $applicant->whatsapp,
      'dob' => $applicant->dob,
      'gender' => $applicant->gender,
      'military_status' => $applicant->military_status,
      'address' => $applicant->address,
      'district' => $applicant->district,
      'governorate' => $applicant->governorate,
      'birth_place' => $applicant->birth_place,
      'degree' => $applicant->degree,
      'major' => $applicant->major,
      'college' => $applicant->college,
      'university' => $applicant->university,
      'is_studying' => $applicant->is_studying,
      'study_year' => $applicant->study_year,
      'grad_year' => $applicant->grad_year,
      'current_job' => $applicant->current_job,
      'employer' => $applicant->employer,
      'experience_public' => $applicant->experience_public,
      'experience_ministry' => $applicant->experience_ministry,
      'experience_other_orgs' => $applicant->experience_other_orgs,
      'visited_countries' => $applicant->visited_countries,
      'center_memberships' => $applicant->center_memberships,
      'club_memberships' => $applicant->club_memberships,
      'sports' => $applicant->sports,
      'skills_hobbies' => $applicant->skills_hobbies,
      'motivation' => $applicant->motivation,
      'strengths' => $applicant->strengths,
      'weaknesses' => $applicant->weaknesses,
      'facebook' => $applicant->facebook,
      'instagram' => $applicant->instagram,
      'linkedin' => $applicant->linkedin,
      'height' => $applicant->height,
      'weight' => $applicant->weight,
      'nationalities' => $applicant->nationalities,
      'languages' => $applicant->languages,
      'degrees' => $applicant->degrees,
      'documents' => $applicant->documents,
      'created_at' => $applicant->created_at,
    ];
  }

  /**
   * Get exam attempt data
   */
  private function getExamData(User $user): ?array
  {
    $attempt = $user->examAttempts()->latest()->first();

    if (!$attempt) {
      return null;
    }

    $exam = $attempt->exam;
    $passingScore = \App\Models\Setting::get('passing_score', 60);

    return [
      'id' => $attempt->id,
      'exam_id' => $attempt->exam_id,
      'exam_title' => $exam->title,
      'exam' => [
        'id' => $exam->id,
        'title' => $exam->title,
        'passing_score' => $passingScore,
        'form_schema' => $exam->form_schema,
      ],
      'score' => $attempt->score,
      'passing_score' => $passingScore,
      'auto_publish_results' => \App\Models\Setting::get('auto_publish_results', true),
      'answers' => $attempt->answers,
      'started_at' => $attempt->started_at,
      'completed_at' => $attempt->completed_at,
      'auto_submitted' => $attempt->auto_submitted,
    ];
  }

  /**
   * Get all evaluations with statistics
   */
  private function getEvaluationsData(User $user): array
  {
    $evaluations = $user->evaluations()
      ->with(['judge', 'evaluationSheet'])
      ->whereNotNull('submitted_at')
      ->get();

    $evaluationsData = $evaluations->map(function ($evaluation) {
      return [
        'id' => $evaluation->id,
        'judge' => [
          'id' => $evaluation->judge->id,
          'name' => $evaluation->judge->full_name,
          'email' => $evaluation->judge->email,
        ],
        'evaluation_sheet' => [
          'id' => $evaluation->evaluationSheet->id,
          'title' => $evaluation->evaluationSheet->title,
        ],
        'answers' => $evaluation->answers,
        'score' => $evaluation->score,
        'submitted_at' => $evaluation->submitted_at,
      ];
    });

    return [
      'count' => $evaluations->count(),
      'max_allowed' => 5,
      'evaluations' => $evaluationsData,
    ];
  }

  /**
   * Get aggregate evaluation data
   */
  private function getAggregateData(User $user): ?array
  {
    $aggregate = $user->evaluationAggregate;

    if (!$aggregate) {
      return null;
    }

    return [
      'total_evaluations' => $aggregate->total_evaluations,
      'criterion_averages' => $aggregate->criterion_averages,
      'overall_score' => $aggregate->overall_score,
      'completed_at' => $aggregate->completed_at,
    ];
  }

  /**
   * Accept a user
   */
  public function acceptUser(User $user, User $reviewer, ?string $notes = null): void
  {
    if ($user->status !== 'evaluated') {
      throw new \Exception('User must be evaluated before being accepted');
    }

    $user->update(['status' => 'accepted']);

    $this->auditService->log(
      causer: $reviewer,
      action: 'user.accepted',
      subject: $user,
      meta: [
        'user_id' => $user->id,
        'user_email' => $user->email,
        'notes' => $notes,
      ]
    );
  }

  /**
   * Reject a user
   */
  public function rejectUser(User $user, User $reviewer, ?string $reason = null): void
  {
    if (!in_array($user->status, ['evaluated', 'under_evaluation', 'exam_completed'])) {
      throw new \Exception('Cannot reject user at this stage');
    }

    $user->update(['status' => 'rejected']);

    $this->auditService->log(
      causer: $reviewer,
      action: 'user.rejected',
      subject: $user,
      meta: [
        'user_id' => $user->id,
        'user_email' => $user->email,
        'reason' => $reason,
      ]
    );
  }

  /**
   * Get statistics for all users or a specific category
   */
  public function getUserStatistics(?int $categoryId = null): array
  {
    $query = User::query();

    if ($categoryId) {
      $query->whereHas('categories', function ($q) use ($categoryId) {
        $q->where('category_id', $categoryId);
      });
    }

    return [
      'total_users' => $query->count(),
      'pending' => (clone $query)->where('status', 'pending')->count(),
      'exam_completed' => (clone $query)->where('status', 'exam_completed')->count(),
      'under_evaluation' => (clone $query)->where('status', 'under_evaluation')->count(),
      'evaluated' => (clone $query)->where('status', 'evaluated')->count(),
      'accepted' => (clone $query)->where('status', 'accepted')->count(),
      'rejected' => (clone $query)->where('status', 'rejected')->count(),
      'total_evaluations' => (clone $query)->where('status', 'evaluated')->count(),
      'avg_evaluation_score' => (clone $query)
        ->where('status', 'evaluated')
        ->join('evaluation_aggregates', 'users.id', '=', 'evaluation_aggregates.evaluated_user_id')
        ->avg('evaluation_aggregates.overall_score'),
    ];
  }
}
