<?php

use Illuminate\Support\Facades\Route;
use Illuminate\Http\Request;
use App\Http\Controllers\Api\V1\IdentifyController;
use App\Http\Controllers\Api\V1\FormController;
use App\Http\Controllers\Api\V1\RegistrationController;
use App\Http\Controllers\Api\V1\RegistrationCheckController;
use App\Http\Controllers\ApplicantController;
use App\Http\Controllers\ExamController;
use App\Http\Controllers\EvaluationController;
use App\Http\Controllers\CategoryController;
use App\Http\Controllers\UserReviewController;
use App\Http\Controllers\SystemLogController;
use App\Http\Controllers\SettingController;
use App\Http\Controllers\AuthController;
use App\Http\Controllers\DocumentController;
use App\Http\Controllers\LanguageController;
use App\Http\Controllers\CountryController;
use App\Http\Controllers\GovernorateController;
use App\Models\Country;
use App\Models\Language;
use App\Models\Governorate;

Route::prefix('v1')->group(function () {

    // ===== AUTHENTICATION ENDPOINTS =====
    Route::post('/login', [AuthController::class, 'login']);
    Route::post('/forgot-password', [AuthController::class, 'forgotPassword']);

    // System status endpoints (not protected by maintenance mode)
    Route::get('/system-status', function () {
        return response()->json([
            'status' => true,
            'registration_open' => \App\Models\Setting::get('registration_open', true),
            'maintenance_mode' => \App\Models\Setting::get('maintenance_mode', false),
        ]);
    });

    // Public endpoints (protected by maintenance mode)
    Route::middleware('maintenance')->group(function () {
        // Legacy endpoint for backward compatibility
        Route::get('/registration-status', function () {
            return response()->json([
                'status' => true,
                'registration_open' => \App\Models\Setting::get('registration_open', true),
            ]);
        });

        // Check if user already applied (by email or national_id)
        Route::post('/check-already-applied', [RegistrationCheckController::class, 'checkAlreadyApplied']);

        // Check if device (IP/fingerprint) has already submitted
        Route::post('/check-device-submission', [RegistrationCheckController::class, 'checkByDeviceFingerprint']);
    });

    // Applicant registration (protected by registration check and maintenance mode)
    Route::middleware(['maintenance', 'registration.open'])->group(function () {
        Route::apiResource('applicants', ApplicantController::class);
    });

    // Reusable dynamic list endpoints
    Route::get('countries', function () {
        return handleResponse(Country::all());
    });

    Route::get('languages', function () {
        return handleResponse(Language::all());
    });

    Route::get('governorates', function () {
        return handleResponse(Governorate::all());
    });

    // Protected endpoints (with maintenance mode check for non-admin users)
    Route::middleware(['auth:sanctum', 'maintenance'])->group(function () {

        // ===== USER PROFILE ENDPOINTS =====
        Route::get('/me', [AuthController::class, 'me']);
        Route::post('/logout', [AuthController::class, 'logout']);
        Route::post('/change-password', [AuthController::class, 'changePassword']);

        // ===== USER ROUTES =====
        Route::get('/applicant', function (Request $request) {
            $applicant = $request->user()->applicant;
            if (!$applicant) {
                return response()->json(['message' => 'No applicant data found'], 404);
            }
            return response()->json([
                'success' => true,
                'data' => $applicant->load(['governorate', 'nationalities', 'languages', 'degrees', 'documents']),
            ]);
        });

        // Document download endpoint
        Route::get('/documents/{document}/download', [DocumentController::class, 'download']);
        Route::get('/documents/{document}/preview', [DocumentController::class, 'preview']);
        Route::get('/documents/{document}', [DocumentController::class, 'show']);

        // ===== EXAM ROUTES (User) =====
        Route::prefix('exams')->group(function () {
            Route::get('/', [ExamController::class, 'index']);
            Route::get('/{exam}', [ExamController::class, 'show']);
            Route::get('/{exam}/results', [ExamController::class, 'results']);
            Route::post('/{exam}/start', [ExamController::class, 'start']);
            Route::post('/{exam}/submit', [ExamController::class, 'submit']);
        });

        // ===== JUDGE ROUTES =====
        Route::middleware('role:judge')->prefix('judge')->group(function () {
            // Statistics
            Route::get('/statistics', [EvaluationController::class, 'statistics']);

            // Evaluation sheets (judges can only view active sheets for selection)
            Route::get('/evaluation-sheets', [EvaluationController::class, 'sheets']);

            // Users available for evaluation
            Route::get('/users', [EvaluationController::class, 'availableUsers']);
            Route::get('/available-users', [EvaluationController::class, 'availableUsers']);

            // Evaluations
            Route::post('/evaluations', [EvaluationController::class, 'submit']);
            Route::get('/evaluations', [EvaluationController::class, 'myEvaluations']);

            // View user data for evaluation
            Route::get('/users/{user}', [UserReviewController::class, 'show']);
            Route::get('/users/{user}/applicant-data', [UserReviewController::class, 'applicantData']);
            Route::post('/users/{user}/evaluate', [EvaluationController::class, 'evaluateUser']);
            Route::get('/users/{user}/evaluations', [EvaluationController::class, 'userEvaluations']);
            Route::get('/users/{user}/exam-results', [UserReviewController::class, 'examResults']);

            // Document access for users being evaluated
            Route::get('/users/{userId}/documents/{documentType}/preview', [DocumentController::class, 'previewByType'])->where('documentType', 'photo_.*|cv|national_id');
            Route::get('/users/{user}/documents/{document}/download', [DocumentController::class, 'judgeDownload']);
            Route::get('/users/{user}/documents/{document}/preview', [DocumentController::class, 'judgePreview']);
            Route::get('/users/{user}/documents/{document}', [DocumentController::class, 'judgeShow']);
        });

        // ===== ADMIN/REVIEWER ROUTES =====
        Route::middleware('role:admin')->prefix('admin')->group(function () {

            // Statistics
            Route::get('/statistics', [UserReviewController::class, 'statistics']);

            // Evaluation sheet management (admin only)
            Route::get('/evaluation-sheets', [EvaluationController::class, 'allSheets']);
            Route::post('/evaluation-sheets', [EvaluationController::class, 'createSheet']);
            Route::put('/evaluation-sheets/{sheet}', [EvaluationController::class, 'updateSheet']);
            Route::delete('/evaluation-sheets/{sheet}', [EvaluationController::class, 'deleteSheet']);

            // Exam management
            Route::get('/exams', [ExamController::class, 'adminIndex']);
            Route::get('/exams/{exam}', [ExamController::class, 'adminShow']);
            Route::post('/exams', [ExamController::class, 'store']);
            Route::put('/exams/{exam}', [ExamController::class, 'update']);
            Route::delete('/exams/{exam}', [ExamController::class, 'destroy']);
            Route::post('/exams/{exam}/publish', [ExamController::class, 'publish']);
            Route::post('/exams/{exam}/unpublish', [ExamController::class, 'unpublish']);
            Route::get('/exams/{exam}/attempts', [ExamController::class, 'attempts']);

            // Category management
            Route::apiResource('categories', CategoryController::class)->only(['index', 'store', 'update', 'destroy']);
            Route::get('/categories/{category}/users', [CategoryController::class, 'users']);
            Route::post('/categories/{category}/assign-user', [CategoryController::class, 'assignUser']);
            Route::post('/categories/{category}/remove-user', [CategoryController::class, 'removeUser']);
            Route::post('/categories/{category}/bulk-assign', [CategoryController::class, 'bulkAssignUsers']);
            Route::post('/categories/{category}/bulk-remove', [CategoryController::class, 'bulkRemoveUsers']);

            // User review and management
            Route::get('/users', [UserReviewController::class, 'index']);
            Route::get('/users/statistics', [UserReviewController::class, 'statistics']);
            Route::get('/users/export/excel', [UserReviewController::class, 'exportToExcel']);
            Route::get('/users/{userId}/documents/{documentType}/preview', [DocumentController::class, 'adminPreviewByType'])->where('documentType', 'photo_.*|cv|national_id');
            Route::get('/users/{user}', [UserReviewController::class, 'show']);
            Route::get('/users/{user}/applicant-data', [UserReviewController::class, 'applicantData']);
            Route::get('/users/{user}/exam-attempt', [UserReviewController::class, 'examAttempt']);
            Route::get('/users/{user}/exam-results', [UserReviewController::class, 'examResults']);
            Route::post('/users/{user}/assign-categories', [UserReviewController::class, 'assignCategories']);
            Route::post('/users/{user}/remove-categories', [UserReviewController::class, 'removeCategories']);
            Route::get('/users/{user}/evaluations', [EvaluationController::class, 'userEvaluations']);
            Route::post('/users/{user}/accept', [UserReviewController::class, 'accept']);
            Route::post('/users/{user}/reject', [UserReviewController::class, 'reject']);

            // System logs
            Route::get('/logs', [SystemLogController::class, 'index']);
            Route::get('/logs/actions', [SystemLogController::class, 'actions']);
            Route::get('/logs/statistics', [SystemLogController::class, 'statistics']);
            Route::get('/logs/user/{userId}', [SystemLogController::class, 'userLogs']);

            // Settings
            Route::get('/settings', [SettingController::class, 'index']);
            Route::put('/settings', [SettingController::class, 'update']);
            Route::get('/settings/{key}', [SettingController::class, 'show']);

            // Languages Management
            Route::apiResource('languages', LanguageController::class);

            // Countries Management
            Route::apiResource('countries', CountryController::class);

            // Governorates Management
            Route::apiResource('governorates', GovernorateController::class);
        });
    });
});
