"use client";

import { useEffect, useState } from "react";
import { adminAPI, handleAPIError } from "@/lib/api";
import { Plus, Edit2, Trash2, X, Search, Globe } from "lucide-react";
import { Button } from "@/components/ui/Button";

interface Country {
  id: number;
  name_en: string;
  name_ar: string;
  iso_code?: string;
  created_at: string;
}

export default function CountriesPage() {
  const [countries, setCountries] = useState<Country[]>([]);
  const [filteredCountries, setFilteredCountries] = useState<Country[]>([]);
  const [loading, setLoading] = useState(true);
  const [showModal, setShowModal] = useState(false);
  const [editingCountry, setEditingCountry] = useState<Country | null>(null);
  const [formData, setFormData] = useState({
    name_en: "",
    name_ar: "",
    iso_code: "",
  });
  const [searchTerm, setSearchTerm] = useState("");
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [notification, setNotification] = useState<{
    show: boolean;
    message: string;
    type: "success" | "error";
  }>({ show: false, message: "", type: "success" });

  useEffect(() => {
    loadCountries();
  }, []);

  useEffect(() => {
    if (searchTerm.trim() === "") {
      setFilteredCountries(countries);
    } else {
      const filtered = countries.filter(
        (country) =>
          country.name_en.toLowerCase().includes(searchTerm.toLowerCase()) ||
          country.name_ar.toLowerCase().includes(searchTerm.toLowerCase()) ||
          country.iso_code?.toLowerCase().includes(searchTerm.toLowerCase())
      );
      setFilteredCountries(filtered);
    }
  }, [searchTerm, countries]);

  const showNotification = (message: string, type: "success" | "error") => {
    setNotification({ show: true, message, type });
    setTimeout(() => {
      setNotification({ show: false, message: "", type: "success" });
    }, 3000);
  };

  const loadCountries = async () => {
    try {
      const response = await adminAPI.getCountries();
      setCountries(response.data.data);
      setFilteredCountries(response.data.data);
    } catch (err: any) {
      showNotification(handleAPIError(err), "error");
    } finally {
      setLoading(false);
    }
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();

    if (!formData.name_en.trim() || !formData.name_ar.trim()) {
      showNotification("الاسم الإنجليزي والعربي مطلوبان", "error");
      return;
    }

    setIsSubmitting(true);
    try {
      if (editingCountry) {
        await adminAPI.updateCountry(editingCountry.id, formData);
        showNotification("تم تحديث الدولة بنجاح", "success");
      } else {
        await adminAPI.createCountry(formData);
        showNotification("تم إنشاء الدولة بنجاح", "success");
      }

      setShowModal(false);
      setFormData({ name_en: "", name_ar: "", iso_code: "" });
      setEditingCountry(null);
      loadCountries();
    } catch (err: any) {
      showNotification(handleAPIError(err), "error");
    } finally {
      setIsSubmitting(false);
    }
  };

  const handleEdit = (country: Country) => {
    setEditingCountry(country);
    setFormData({
      name_en: country.name_en,
      name_ar: country.name_ar,
      iso_code: country.iso_code || "",
    });
    setShowModal(true);
  };

  const handleDelete = async (id: number) => {
    if (confirm("هل أنت متأكد من حذف هذه الدولة؟")) {
      try {
        await adminAPI.deleteCountry(id);
        showNotification("تم حذف الدولة بنجاح", "success");
        loadCountries();
      } catch (err: any) {
        showNotification(handleAPIError(err), "error");
      }
    }
  };

  const closeModal = () => {
    setShowModal(false);
    setFormData({ name_en: "", name_ar: "", iso_code: "" });
    setEditingCountry(null);
  };

  return (
    <div className="space-y-6" dir="rtl">
      {/* Notification */}
      {notification.show && (
        <div
          className={`p-4 rounded-lg ${
            notification.type === "success"
              ? "bg-green-50 text-green-800 border border-green-200"
              : "bg-red-50 text-red-800 border border-red-200"
          }`}
        >
          {notification.message}
        </div>
      )}

      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <div className="flex items-center gap-2">
            <Globe className="w-8 h-8 text-blue-600" />
            <h1 className="text-3xl font-bold text-gray-900">الدول</h1>
          </div>
          <p className="text-gray-600 mt-1">إدارة الدول</p>
        </div>
        <Button
          onClick={() => setShowModal(true)}
          className="bg-blue-600 hover:bg-blue-700 text-white flex items-center gap-2"
        >
          <Plus className="w-4 h-4" />
          إضافة دولة
        </Button>
      </div>

      {/* Search */}
      <div className="relative">
        <Search className="absolute right-3 top-3 w-5 h-5 text-gray-400" />
        <input
          type="text"
          placeholder="ابحث عن دولة..."
          value={searchTerm}
          onChange={(e) => setSearchTerm(e.target.value)}
          className="w-full pr-10 pl-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent text-right"
        />
      </div>

      {/* Countries Table */}
      <div className="bg-white rounded-lg shadow overflow-hidden">
        {loading ? (
          <div className="flex items-center justify-center h-64">
            <div className="text-gray-500">جاري التحميل...</div>
          </div>
        ) : filteredCountries.length === 0 ? (
          <div className="flex items-center justify-center h-64">
            <div className="text-gray-500">لم يتم العثور على دول</div>
          </div>
        ) : (
          <table className="w-full">
            <thead className="bg-gray-50 border-b border-gray-200">
              <tr>
                <th className="px-6 py-3 text-right text-sm font-medium text-gray-700">
                  الاسم الإنجليزي
                </th>
                <th className="px-6 py-3 text-right text-sm font-medium text-gray-700">
                  الاسم العربي
                </th>
                <th className="px-6 py-3 text-right text-sm font-medium text-gray-700">
                  رمز ISO
                </th>
                <th className="px-6 py-3 text-right text-sm font-medium text-gray-700">
                  الإجراءات
                </th>
              </tr>
            </thead>
            <tbody>
              {filteredCountries.map((country) => (
                <tr
                  key={country.id}
                  className="border-b border-gray-200 hover:bg-gray-50"
                >
                  <td className="px-6 py-4 text-sm text-gray-900">
                    {country.name_en}
                  </td>
                  <td className="px-6 py-4 text-sm text-gray-900 text-right">
                    {country.name_ar}
                  </td>
                  <td className="px-6 py-4 text-sm text-gray-600">
                    {country.iso_code || "-"}
                  </td>
                  <td className="px-6 py-4 text-sm">
                    <div className="flex items-center gap-2">
                      <button
                        onClick={() => handleEdit(country)}
                        className="p-2 text-blue-600 hover:bg-blue-50 rounded"
                      >
                        <Edit2 className="w-4 h-4" />
                      </button>
                      <button
                        onClick={() => handleDelete(country.id)}
                        className="p-2 text-red-600 hover:bg-red-50 rounded"
                      >
                        <Trash2 className="w-4 h-4" />
                      </button>
                    </div>
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        )}
      </div>

      {/* Modal */}
      {showModal && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
          <div
            className="bg-white rounded-lg shadow-lg p-6 w-full max-w-md"
            dir="rtl"
          >
            <div className="flex items-center justify-between mb-4">
              <h2 className="text-xl font-bold">
                {editingCountry ? "تعديل الدولة" : "إضافة دولة جديدة"}
              </h2>
              <button
                onClick={closeModal}
                className="text-gray-500 hover:text-gray-700"
              >
                <X className="w-5 h-5" />
              </button>
            </div>

            <form onSubmit={handleSubmit} className="space-y-4">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">
                  الاسم الإنجليزي *
                </label>
                <input
                  type="text"
                  value={formData.name_en}
                  onChange={(e) =>
                    setFormData({ ...formData, name_en: e.target.value })
                  }
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  placeholder="مثل: مصر"
                />
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">
                  الاسم العربي *
                </label>
                <input
                  type="text"
                  value={formData.name_ar}
                  onChange={(e) =>
                    setFormData({ ...formData, name_ar: e.target.value })
                  }
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent text-right"
                  placeholder="مثل: مصر"
                />
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">
                  رمز ISO (اختياري)
                </label>
                <input
                  type="text"
                  value={formData.iso_code}
                  onChange={(e) =>
                    setFormData({ ...formData, iso_code: e.target.value })
                  }
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  placeholder="مثل: EG"
                />
              </div>

              <div className="flex gap-3 pt-4">
                <button
                  type="button"
                  onClick={closeModal}
                  className="flex-1 px-4 py-2 border border-gray-300 rounded-lg text-gray-700 hover:bg-gray-50"
                >
                  إلغاء
                </button>
                <button
                  type="submit"
                  disabled={isSubmitting}
                  className="flex-1 px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 disabled:opacity-50"
                >
                  {isSubmitting ? "جاري الحفظ..." : "حفظ"}
                </button>
              </div>
            </form>
          </div>
        </div>
      )}
    </div>
  );
}
