"use client";

import { useEffect, useState } from "react";
import { adminAPI, handleAPIError } from "@/lib/api";
import { Button } from "@/components/ui/Button";
import {
  Plus,
  Edit,
  Trash2,
  Eye,
  CheckCircle,
  XCircle,
  Calendar,
  Users,
  FileText,
} from "lucide-react";

interface Exam {
  id: number;
  title: string;
  description: string;
  form_schema: any;
  is_published: boolean;
  published_at: string | null;
  deadline_at: string | null;
  max_attempts: number;
  attempts_count: number;
  created_at: string;
}

interface Question {
  id: string;
  text: string;
  type: "multiple_choice" | "true_false" | "essay";
  options: string[];
  correct_answer: string;
  points: number;
}

export default function ExamsPage() {
  const [exams, setExams] = useState<Exam[]>([]);
  const [loading, setLoading] = useState(true);
  const [showCreateModal, setShowCreateModal] = useState(false);
  const [showEditModal, setShowEditModal] = useState(false);
  const [selectedExam, setSelectedExam] = useState<Exam | null>(null);
  const [formData, setFormData] = useState({
    title: "",
    description: "",
    deadline_at: "",
  });
  const [questions, setQuestions] = useState<Question[]>([]);

  useEffect(() => {
    loadExams();
  }, []);

  const loadExams = async () => {
    setLoading(true);
    try {
      const response = await adminAPI.getExams();
      setExams(response.data.data);
    } catch (err: any) {
      console.error(handleAPIError(err));
    } finally {
      setLoading(false);
    }
  };

  const addQuestion = () => {
    const newQuestion: Question = {
      id: Date.now().toString(),
      text: "",
      type: "multiple_choice",
      options: ["", "", "", ""],
      correct_answer: "",
      points: 1,
    };
    setQuestions([...questions, newQuestion]);
  };

  const updateQuestion = (id: string, field: keyof Question, value: any) => {
    setQuestions(
      questions.map((q) => (q.id === id ? { ...q, [field]: value } : q))
    );
  };

  const updateQuestionOption = (
    questionId: string,
    optionIndex: number,
    value: string
  ) => {
    setQuestions(
      questions.map((q) => {
        if (q.id === questionId) {
          const newOptions = [...q.options];
          newOptions[optionIndex] = value;
          return { ...q, options: newOptions };
        }
        return q;
      })
    );
  };

  const addQuestionOption = (questionId: string) => {
    setQuestions(
      questions.map((q) => {
        if (q.id === questionId) {
          return { ...q, options: [...q.options, ""] };
        }
        return q;
      })
    );
  };

  const removeQuestionOption = (questionId: string, optionIndex: number) => {
    setQuestions(
      questions.map((q) => {
        if (q.id === questionId) {
          const newOptions = q.options.filter((_, i) => i !== optionIndex);
          return { ...q, options: newOptions };
        }
        return q;
      })
    );
  };

  const removeQuestion = (id: string) => {
    setQuestions(questions.filter((q) => q.id !== id));
  };

  const handleCreate = async (e: React.FormEvent) => {
    e.preventDefault();

    if (questions.length === 0) {
      alert("يجب إضافة سؤال واحد على الأقل");
      return;
    }

    // Validate questions
    for (const q of questions) {
      if (!q.text.trim()) {
        alert("يجب إدخال نص السؤال");
        return;
      }
      if (
        q.type === "multiple_choice" &&
        q.options.filter((o) => o.trim()).length < 2
      ) {
        alert("يجب إضافة خيارين على الأقل لأسئلة الاختيار من متعدد");
        return;
      }
      if (!q.correct_answer.trim()) {
        alert("يجب تحديد الإجابة الصحيحة لكل سؤال");
        return;
      }
    }

    try {
      const schema = {
        questions: questions.map((q, index) => ({
          id: index + 1,
          text: q.text,
          type: q.type,
          options: q.type === "essay" ? [] : q.options.filter((o) => o.trim()),
          correct_answer: q.correct_answer,
          points: q.points,
        })),
      };

      await adminAPI.createExam({
        title: formData.title,
        description: formData.description,
        deadline_at: formData.deadline_at || null,
        form_schema: schema,
      });

      await loadExams();
      setShowCreateModal(false);
      setFormData({ title: "", description: "", deadline_at: "" });
      setQuestions([]);
      alert("تم إنشاء الاختبار بنجاح");
    } catch (err: any) {
      alert(handleAPIError(err));
    }
  };

  const handleEdit = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!selectedExam) return;

    if (questions.length === 0) {
      alert("يجب إضافة سؤال واحد على الأقل");
      return;
    }

    // Validate questions
    for (const q of questions) {
      if (!q.text.trim()) {
        alert("يجب إدخال نص السؤال");
        return;
      }
      if (
        q.type === "multiple_choice" &&
        q.options.filter((o) => o.trim()).length < 2
      ) {
        alert("يجب إضافة خيارين على الأقل لأسئلة الاختيار من متعدد");
        return;
      }
      if (!q.correct_answer.trim()) {
        alert("يجب تحديد الإجابة الصحيحة لكل سؤال");
        return;
      }
    }

    try {
      const schema = {
        questions: questions.map((q, index) => ({
          id: index + 1,
          text: q.text,
          type: q.type,
          options: q.type === "essay" ? [] : q.options.filter((o) => o.trim()),
          correct_answer: q.correct_answer,
          points: q.points,
        })),
      };

      await adminAPI.updateExam(selectedExam.id, {
        title: formData.title,
        description: formData.description,
        deadline_at: formData.deadline_at || null,
        form_schema: schema,
      });

      await loadExams();
      setShowEditModal(false);
      setSelectedExam(null);
      setQuestions([]);
      alert("تم تحديث الاختبار بنجاح");
    } catch (err: any) {
      alert(handleAPIError(err));
    }
  };

  const handleDelete = async (exam: Exam) => {
    if (!confirm(`هل أنت متأكد من حذف الاختبار "${exam.title}"؟`)) return;

    try {
      await adminAPI.deleteExam(exam.id);
      await loadExams();
      alert("تم حذف الاختبار بنجاح");
    } catch (err: any) {
      alert(handleAPIError(err));
    }
  };

  const handlePublish = async (exam: Exam) => {
    try {
      if (exam.is_published) {
        await adminAPI.unpublishExam(exam.id);
        alert("تم إلغاء نشر الاختبار");
      } else {
        await adminAPI.publishExam(exam.id);
        alert("تم نشر الاختبار بنجاح");
      }
      await loadExams();
    } catch (err: any) {
      alert(handleAPIError(err));
    }
  };

  const openEditModal = (exam: Exam) => {
    setSelectedExam(exam);
    setFormData({
      title: exam.title,
      description: exam.description || "",
      deadline_at: exam.deadline_at ? exam.deadline_at.split("T")[0] : "",
    });

    // Parse existing questions from form_schema
    if (exam.form_schema?.questions) {
      const existingQuestions: Question[] = exam.form_schema.questions.map(
        (q: any) => ({
          id: q.id.toString(),
          text: q.text,
          type: q.type,
          options: q.options || ["", "", "", ""],
          correct_answer: q.correct_answer,
          points: q.points || 1,
        })
      );
      setQuestions(existingQuestions);
    } else {
      setQuestions([]);
    }

    setShowEditModal(true);
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center h-screen">
        <div className="w-12 h-12 border-4 border-blue-600 border-t-transparent rounded-full animate-spin" />
      </div>
    );
  }

  return (
    <div className="p-8">
      {/* Header */}
      <div className="flex items-center justify-between mb-8">
        <div>
          <h1 className="text-3xl font-bold text-gray-900">إدارة الاختبارات</h1>
          <p className="text-gray-600 mt-2">إنشاء وإدارة اختبارات المتقدمين</p>
        </div>
        <Button onClick={() => setShowCreateModal(true)}>
          <Plus className="w-5 h-5 ml-2" />
          إنشاء اختبار جديد
        </Button>
      </div>

      {/* Stats */}
      <div className="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
        <div className="bg-gradient-to-br from-blue-50 to-blue-100 border border-blue-200 rounded-lg p-6">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-blue-600 font-medium">
                إجمالي الاختبارات
              </p>
              <p className="text-3xl font-bold text-blue-900 mt-2">
                {exams.length}
              </p>
            </div>
            <div className="p-3 bg-blue-200 rounded-lg">
              <FileText className="w-8 h-8 text-blue-700" />
            </div>
          </div>
        </div>

        <div className="bg-gradient-to-br from-green-50 to-green-100 border border-green-200 rounded-lg p-6">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-green-600 font-medium">
                الاختبارات المنشورة
              </p>
              <p className="text-3xl font-bold text-green-900 mt-2">
                {exams.filter((e) => e.is_published).length}
              </p>
            </div>
            <div className="p-3 bg-green-200 rounded-lg">
              <CheckCircle className="w-8 h-8 text-green-700" />
            </div>
          </div>
        </div>

        <div className="bg-gradient-to-br from-purple-50 to-purple-100 border border-purple-200 rounded-lg p-6">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-purple-600 font-medium">
                إجمالي المحاولات
              </p>
              <p className="text-3xl font-bold text-purple-900 mt-2">
                {exams.reduce((sum, e) => sum + e.attempts_count, 0)}
              </p>
            </div>
            <div className="p-3 bg-purple-200 rounded-lg">
              <Users className="w-8 h-8 text-purple-700" />
            </div>
          </div>
        </div>
      </div>

      {/* Exams List */}
      <div className="bg-white rounded-lg shadow">
        {exams.length === 0 ? (
          <div className="text-center py-12">
            <FileText className="w-16 h-16 text-gray-400 mx-auto mb-4" />
            <p className="text-gray-500">لا توجد اختبارات حتى الآن</p>
            <Button onClick={() => setShowCreateModal(true)} className="mt-4">
              إنشاء أول اختبار
            </Button>
          </div>
        ) : (
          <div className="overflow-x-auto">
            <table className="w-full">
              <thead className="bg-gray-50 border-b border-gray-200">
                <tr>
                  <th className="px-6 py-4 text-right text-sm font-semibold text-gray-900">
                    العنوان
                  </th>
                  <th className="px-6 py-4 text-right text-sm font-semibold text-gray-900">
                    الحالة
                  </th>
                  <th className="px-6 py-4 text-right text-sm font-semibold text-gray-900">
                    الموعد النهائي
                  </th>
                  <th className="px-6 py-4 text-right text-sm font-semibold text-gray-900">
                    المحاولات
                  </th>
                  <th className="px-6 py-4 text-right text-sm font-semibold text-gray-900">
                    الإجراءات
                  </th>
                </tr>
              </thead>
              <tbody className="divide-y divide-gray-200">
                {exams.map((exam) => (
                  <tr key={exam.id} className="hover:bg-gray-50">
                    <td className="px-6 py-4">
                      <div>
                        <p className="font-semibold text-gray-900">
                          {exam.title}
                        </p>
                        {exam.description && (
                          <p className="text-sm text-gray-600 mt-1 line-clamp-1">
                            {exam.description}
                          </p>
                        )}
                      </div>
                    </td>
                    <td className="px-6 py-4">
                      <span
                        className={`inline-flex items-center gap-2 px-3 py-1 rounded-full text-sm font-medium ${
                          exam.is_published
                            ? "bg-green-100 text-green-800"
                            : "bg-gray-100 text-gray-800"
                        }`}
                      >
                        {exam.is_published ? (
                          <>
                            <CheckCircle className="w-4 h-4" />
                            منشور
                          </>
                        ) : (
                          <>
                            <XCircle className="w-4 h-4" />
                            مسودة
                          </>
                        )}
                      </span>
                    </td>
                    <td className="px-6 py-4">
                      {exam.deadline_at ? (
                        <div className="flex items-center gap-2 text-sm text-gray-600">
                          <Calendar className="w-4 h-4" />
                          {new Date(exam.deadline_at).toLocaleDateString(
                            "ar-EG"
                          )}
                        </div>
                      ) : (
                        <span className="text-gray-400">-</span>
                      )}
                    </td>
                    <td className="px-6 py-4">
                      <div className="flex items-center gap-2">
                        <Users className="w-4 h-4 text-gray-600" />
                        <span className="font-semibold text-gray-900">
                          {exam.attempts_count}
                        </span>
                      </div>
                    </td>
                    <td className="px-6 py-4">
                      <div className="flex items-center gap-2">
                        <button
                          onClick={() => handlePublish(exam)}
                          className="p-2 hover:bg-gray-100 rounded-lg transition-colors"
                          title={exam.is_published ? "إلغاء النشر" : "نشر"}
                        >
                          {exam.is_published ? (
                            <XCircle className="w-5 h-5 text-orange-600" />
                          ) : (
                            <CheckCircle className="w-5 h-5 text-green-600" />
                          )}
                        </button>
                        <button
                          onClick={() => openEditModal(exam)}
                          className="p-2 hover:bg-gray-100 rounded-lg transition-colors"
                          title="تعديل"
                        >
                          <Edit className="w-5 h-5 text-blue-600" />
                        </button>
                        <button
                          onClick={() => handleDelete(exam)}
                          className="p-2 hover:bg-gray-100 rounded-lg transition-colors"
                          title="حذف"
                          disabled={exam.attempts_count > 0}
                        >
                          <Trash2
                            className={`w-5 h-5 ${
                              exam.attempts_count > 0
                                ? "text-gray-400"
                                : "text-red-600"
                            }`}
                          />
                        </button>
                      </div>
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        )}
      </div>

      {/* Create Modal */}
      {showCreateModal && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center p-4 z-50">
          <div className="bg-white rounded-lg max-w-4xl w-full max-h-[90vh] overflow-y-auto">
            <div className="p-6 border-b border-gray-200">
              <h2 className="text-2xl font-bold text-gray-900">
                إنشاء اختبار جديد
              </h2>
            </div>
            <form onSubmit={handleCreate} className="p-6 space-y-6">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  عنوان الاختبار *
                </label>
                <input
                  type="text"
                  required
                  value={formData.title}
                  onChange={(e) =>
                    setFormData({ ...formData, title: e.target.value })
                  }
                  className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                />
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  الوصف
                </label>
                <textarea
                  value={formData.description}
                  onChange={(e) =>
                    setFormData({ ...formData, description: e.target.value })
                  }
                  rows={3}
                  className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                />
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  الموعد النهائي
                </label>
                <input
                  type="date"
                  value={formData.deadline_at}
                  onChange={(e) =>
                    setFormData({ ...formData, deadline_at: e.target.value })
                  }
                  className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                />
              </div>

              {/* Questions Section */}
              <div className="border-t border-gray-200 pt-6">
                <div className="flex items-center justify-between mb-4">
                  <h3 className="text-lg font-semibold text-gray-900">
                    الأسئلة
                  </h3>
                  <Button type="button" onClick={addQuestion} variant="outline">
                    <Plus className="w-4 h-4 ml-2" />
                    إضافة سؤال
                  </Button>
                </div>

                {questions.length === 0 ? (
                  <div className="text-center py-8 bg-gray-50 rounded-lg">
                    <FileText className="w-12 h-12 text-gray-400 mx-auto mb-2" />
                    <p className="text-gray-500">لم يتم إضافة أي أسئلة بعد</p>
                  </div>
                ) : (
                  <div className="space-y-4">
                    {questions.map((question, qIndex) => (
                      <div
                        key={question.id}
                        className="border border-gray-300 rounded-lg p-4 bg-gray-50"
                      >
                        <div className="flex items-start justify-between mb-3">
                          <h4 className="font-medium text-gray-900">
                            السؤال {qIndex + 1}
                          </h4>
                          <button
                            type="button"
                            onClick={() => removeQuestion(question.id)}
                            className="text-red-600 hover:text-red-700"
                          >
                            <Trash2 className="w-5 h-5" />
                          </button>
                        </div>

                        <div className="space-y-3">
                          <div>
                            <label className="block text-sm font-medium text-gray-700 mb-1">
                              نص السؤال *
                            </label>
                            <input
                              type="text"
                              value={question.text}
                              onChange={(e) =>
                                updateQuestion(
                                  question.id,
                                  "text",
                                  e.target.value
                                )
                              }
                              className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                              placeholder="أدخل نص السؤال"
                            />
                          </div>

                          <div className="grid grid-cols-2 gap-3">
                            <div>
                              <label className="block text-sm font-medium text-gray-700 mb-1">
                                نوع السؤال
                              </label>
                              <select
                                value={question.type}
                                onChange={(e) =>
                                  updateQuestion(
                                    question.id,
                                    "type",
                                    e.target.value
                                  )
                                }
                                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                              >
                                <option value="multiple_choice">
                                  اختيار من متعدد
                                </option>
                                <option value="true_false">صح/خطأ</option>
                                <option value="essay">مقالي</option>
                              </select>
                            </div>

                            <div>
                              <label className="block text-sm font-medium text-gray-700 mb-1">
                                النقاط
                              </label>
                              <input
                                type="number"
                                min="1"
                                value={question.points}
                                onChange={(e) =>
                                  updateQuestion(
                                    question.id,
                                    "points",
                                    parseInt(e.target.value)
                                  )
                                }
                                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                              />
                            </div>
                          </div>

                          {question.type === "multiple_choice" && (
                            <div>
                              <div className="flex items-center justify-between mb-2">
                                <label className="block text-sm font-medium text-gray-700">
                                  الخيارات *
                                </label>
                                <button
                                  type="button"
                                  onClick={() => addQuestionOption(question.id)}
                                  className="text-sm text-blue-600 hover:text-blue-700"
                                >
                                  + إضافة خيار
                                </button>
                              </div>
                              <div className="space-y-2">
                                {question.options.map((option, optIndex) => (
                                  <div
                                    key={optIndex}
                                    className="flex items-center gap-2"
                                  >
                                    <span className="text-sm text-gray-600 w-6">
                                      {String.fromCharCode(65 + optIndex)}.
                                    </span>
                                    <input
                                      type="text"
                                      value={option}
                                      onChange={(e) =>
                                        updateQuestionOption(
                                          question.id,
                                          optIndex,
                                          e.target.value
                                        )
                                      }
                                      className="flex-1 px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                                      placeholder={`الخيار ${String.fromCharCode(
                                        65 + optIndex
                                      )}`}
                                    />
                                    {question.options.length > 2 && (
                                      <button
                                        type="button"
                                        onClick={() =>
                                          removeQuestionOption(
                                            question.id,
                                            optIndex
                                          )
                                        }
                                        className="text-red-600 hover:text-red-700"
                                      >
                                        <XCircle className="w-5 h-5" />
                                      </button>
                                    )}
                                  </div>
                                ))}
                              </div>
                            </div>
                          )}

                          {question.type === "true_false" && (
                            <div>
                              <label className="block text-sm font-medium text-gray-700 mb-2">
                                الإجابة الصحيحة *
                              </label>
                              <div className="flex gap-4">
                                <label className="flex items-center">
                                  <input
                                    type="radio"
                                    name={`correct-${question.id}`}
                                    value="صح"
                                    checked={question.correct_answer === "صح"}
                                    onChange={(e) =>
                                      updateQuestion(
                                        question.id,
                                        "correct_answer",
                                        e.target.value
                                      )
                                    }
                                    className="ml-2"
                                  />
                                  صح
                                </label>
                                <label className="flex items-center">
                                  <input
                                    type="radio"
                                    name={`correct-${question.id}`}
                                    value="خطأ"
                                    checked={question.correct_answer === "خطأ"}
                                    onChange={(e) =>
                                      updateQuestion(
                                        question.id,
                                        "correct_answer",
                                        e.target.value
                                      )
                                    }
                                    className="ml-2"
                                  />
                                  خطأ
                                </label>
                              </div>
                            </div>
                          )}

                          {question.type === "multiple_choice" && (
                            <div>
                              <label className="block text-sm font-medium text-gray-700 mb-1">
                                الإجابة الصحيحة *
                              </label>
                              <select
                                value={question.correct_answer}
                                onChange={(e) =>
                                  updateQuestion(
                                    question.id,
                                    "correct_answer",
                                    e.target.value
                                  )
                                }
                                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                              >
                                <option value="">اختر الإجابة الصحيحة</option>
                                {question.options
                                  .filter((o) => o.trim())
                                  .map((option, idx) => (
                                    <option key={idx} value={option}>
                                      {String.fromCharCode(65 + idx)}. {option}
                                    </option>
                                  ))}
                              </select>
                            </div>
                          )}

                          {question.type === "essay" && (
                            <div>
                              <label className="block text-sm font-medium text-gray-700 mb-1">
                                نموذج الإجابة
                              </label>
                              <textarea
                                value={question.correct_answer}
                                onChange={(e) =>
                                  updateQuestion(
                                    question.id,
                                    "correct_answer",
                                    e.target.value
                                  )
                                }
                                rows={3}
                                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                                placeholder="نموذج الإجابة المتوقعة (اختياري)"
                              />
                            </div>
                          )}
                        </div>
                      </div>
                    ))}
                  </div>
                )}
              </div>

              <div className="flex gap-4 pt-4 border-t border-gray-200">
                <Button type="submit" className="flex-1">
                  إنشاء الاختبار
                </Button>
                <Button
                  type="button"
                  variant="outline"
                  onClick={() => {
                    setShowCreateModal(false);
                    setFormData({
                      title: "",
                      description: "",
                      deadline_at: "",
                    });
                    setQuestions([]);
                  }}
                  className="flex-1"
                >
                  إلغاء
                </Button>
              </div>
            </form>
          </div>
        </div>
      )}

      {/* Edit Modal */}
      {showEditModal && selectedExam && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center p-4 z-50">
          <div className="bg-white rounded-lg max-w-4xl w-full max-h-[90vh] overflow-y-auto">
            <div className="p-6 border-b border-gray-200">
              <h2 className="text-2xl font-bold text-gray-900">
                تعديل الاختبار
              </h2>
            </div>
            <form onSubmit={handleEdit} className="p-6 space-y-6">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  عنوان الاختبار *
                </label>
                <input
                  type="text"
                  required
                  value={formData.title}
                  onChange={(e) =>
                    setFormData({ ...formData, title: e.target.value })
                  }
                  className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                />
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  الوصف
                </label>
                <textarea
                  value={formData.description}
                  onChange={(e) =>
                    setFormData({ ...formData, description: e.target.value })
                  }
                  rows={3}
                  className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                />
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  الموعد النهائي
                </label>
                <input
                  type="date"
                  value={formData.deadline_at}
                  onChange={(e) =>
                    setFormData({ ...formData, deadline_at: e.target.value })
                  }
                  className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                />
              </div>

              {/* Questions Section */}
              <div className="border-t border-gray-200 pt-6">
                <div className="flex items-center justify-between mb-4">
                  <h3 className="text-lg font-semibold text-gray-900">
                    الأسئلة
                  </h3>
                  <Button type="button" onClick={addQuestion} variant="outline">
                    <Plus className="w-4 h-4 ml-2" />
                    إضافة سؤال
                  </Button>
                </div>

                {questions.length === 0 ? (
                  <div className="text-center py-8 bg-gray-50 rounded-lg">
                    <FileText className="w-12 h-12 text-gray-400 mx-auto mb-2" />
                    <p className="text-gray-500">لم يتم إضافة أي أسئلة بعد</p>
                  </div>
                ) : (
                  <div className="space-y-4">
                    {questions.map((question, qIndex) => (
                      <div
                        key={question.id}
                        className="border border-gray-300 rounded-lg p-4 bg-gray-50"
                      >
                        <div className="flex items-start justify-between mb-3">
                          <h4 className="font-medium text-gray-900">
                            السؤال {qIndex + 1}
                          </h4>
                          <button
                            type="button"
                            onClick={() => removeQuestion(question.id)}
                            className="text-red-600 hover:text-red-700"
                          >
                            <Trash2 className="w-5 h-5" />
                          </button>
                        </div>

                        <div className="space-y-3">
                          <div>
                            <label className="block text-sm font-medium text-gray-700 mb-1">
                              نص السؤال *
                            </label>
                            <input
                              type="text"
                              value={question.text}
                              onChange={(e) =>
                                updateQuestion(
                                  question.id,
                                  "text",
                                  e.target.value
                                )
                              }
                              className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                              placeholder="أدخل نص السؤال"
                            />
                          </div>

                          <div className="grid grid-cols-2 gap-3">
                            <div>
                              <label className="block text-sm font-medium text-gray-700 mb-1">
                                نوع السؤال
                              </label>
                              <select
                                value={question.type}
                                onChange={(e) =>
                                  updateQuestion(
                                    question.id,
                                    "type",
                                    e.target.value
                                  )
                                }
                                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                              >
                                <option value="multiple_choice">
                                  اختيار من متعدد
                                </option>
                                <option value="true_false">صح/خطأ</option>
                                <option value="essay">مقالي</option>
                              </select>
                            </div>

                            <div>
                              <label className="block text-sm font-medium text-gray-700 mb-1">
                                النقاط
                              </label>
                              <input
                                type="number"
                                min="1"
                                value={question.points}
                                onChange={(e) =>
                                  updateQuestion(
                                    question.id,
                                    "points",
                                    parseInt(e.target.value)
                                  )
                                }
                                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                              />
                            </div>
                          </div>

                          {question.type === "multiple_choice" && (
                            <div>
                              <div className="flex items-center justify-between mb-2">
                                <label className="block text-sm font-medium text-gray-700">
                                  الخيارات *
                                </label>
                                <button
                                  type="button"
                                  onClick={() => addQuestionOption(question.id)}
                                  className="text-sm text-blue-600 hover:text-blue-700"
                                >
                                  + إضافة خيار
                                </button>
                              </div>
                              <div className="space-y-2">
                                {question.options.map((option, optIndex) => (
                                  <div
                                    key={optIndex}
                                    className="flex items-center gap-2"
                                  >
                                    <span className="text-sm text-gray-600 w-6">
                                      {String.fromCharCode(65 + optIndex)}.
                                    </span>
                                    <input
                                      type="text"
                                      value={option}
                                      onChange={(e) =>
                                        updateQuestionOption(
                                          question.id,
                                          optIndex,
                                          e.target.value
                                        )
                                      }
                                      className="flex-1 px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                                      placeholder={`الخيار ${String.fromCharCode(
                                        65 + optIndex
                                      )}`}
                                    />
                                    {question.options.length > 2 && (
                                      <button
                                        type="button"
                                        onClick={() =>
                                          removeQuestionOption(
                                            question.id,
                                            optIndex
                                          )
                                        }
                                        className="text-red-600 hover:text-red-700"
                                      >
                                        <XCircle className="w-5 h-5" />
                                      </button>
                                    )}
                                  </div>
                                ))}
                              </div>
                            </div>
                          )}

                          {question.type === "true_false" && (
                            <div>
                              <label className="block text-sm font-medium text-gray-700 mb-2">
                                الإجابة الصحيحة *
                              </label>
                              <div className="flex gap-4">
                                <label className="flex items-center">
                                  <input
                                    type="radio"
                                    name={`correct-${question.id}`}
                                    value="صح"
                                    checked={question.correct_answer === "صح"}
                                    onChange={(e) =>
                                      updateQuestion(
                                        question.id,
                                        "correct_answer",
                                        e.target.value
                                      )
                                    }
                                    className="ml-2"
                                  />
                                  صح
                                </label>
                                <label className="flex items-center">
                                  <input
                                    type="radio"
                                    name={`correct-${question.id}`}
                                    value="خطأ"
                                    checked={question.correct_answer === "خطأ"}
                                    onChange={(e) =>
                                      updateQuestion(
                                        question.id,
                                        "correct_answer",
                                        e.target.value
                                      )
                                    }
                                    className="ml-2"
                                  />
                                  خطأ
                                </label>
                              </div>
                            </div>
                          )}

                          {question.type === "multiple_choice" && (
                            <div>
                              <label className="block text-sm font-medium text-gray-700 mb-1">
                                الإجابة الصحيحة *
                              </label>
                              <select
                                value={question.correct_answer}
                                onChange={(e) =>
                                  updateQuestion(
                                    question.id,
                                    "correct_answer",
                                    e.target.value
                                  )
                                }
                                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                              >
                                <option value="">اختر الإجابة الصحيحة</option>
                                {question.options
                                  .filter((o) => o.trim())
                                  .map((option, idx) => (
                                    <option key={idx} value={option}>
                                      {String.fromCharCode(65 + idx)}. {option}
                                    </option>
                                  ))}
                              </select>
                            </div>
                          )}

                          {question.type === "essay" && (
                            <div>
                              <label className="block text-sm font-medium text-gray-700 mb-1">
                                نموذج الإجابة
                              </label>
                              <textarea
                                value={question.correct_answer}
                                onChange={(e) =>
                                  updateQuestion(
                                    question.id,
                                    "correct_answer",
                                    e.target.value
                                  )
                                }
                                rows={3}
                                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                                placeholder="نموذج الإجابة المتوقعة (اختياري)"
                              />
                            </div>
                          )}
                        </div>
                      </div>
                    ))}
                  </div>
                )}
              </div>

              <div className="flex gap-4 pt-4 border-t border-gray-200">
                <Button type="submit" className="flex-1">
                  حفظ التعديلات
                </Button>
                <Button
                  type="button"
                  variant="outline"
                  onClick={() => {
                    setShowEditModal(false);
                    setSelectedExam(null);
                    setQuestions([]);
                  }}
                  className="flex-1"
                >
                  إلغاء
                </Button>
              </div>
            </form>
          </div>
        </div>
      )}
    </div>
  );
}
