"use client";

import { useEffect, useState } from "react";
import { adminAPI, handleAPIError } from "@/lib/api";
import { Button } from "@/components/ui/Button";
import {
  Save,
  Mail,
  Globe,
  Shield,
  Bell,
  Database,
  FileText,
} from "lucide-react";

interface Settings {
  registration_open: boolean;
  exam_duration: number;
  passing_score: number;
  max_login_attempts: number;
  auto_publish_results: boolean;
  maintenance_mode: boolean;
}

export default function SettingsPage() {
  const [settings, setSettings] = useState<Settings>({
    registration_open: true,
    exam_duration: 60,
    passing_score: 50,
    max_login_attempts: 5,
    auto_publish_results: false,
    maintenance_mode: false,
  });
  const [loading, setLoading] = useState(true);
  const [saving, setSaving] = useState(false);
  const [message, setMessage] = useState<{
    type: "success" | "error";
    text: string;
  } | null>(null);

  useEffect(() => {
    loadSettings();
  }, []);

  const loadSettings = async () => {
    setLoading(true);
    try {
      const response = await adminAPI.getSettings();
      if (response.data.data) {
        setSettings((prev) => ({ ...prev, ...response.data.data }));
      }
    } catch (error) {
      console.error("Failed to load settings:", error);
    } finally {
      setLoading(false);
    }
  };

  const handleChange = (field: keyof Settings, value: any) => {
    setSettings((prev) => ({ ...prev, [field]: value }));
  };

  const handleSave = async () => {
    setSaving(true);
    setMessage(null);
    try {
      const response = await adminAPI.updateSettings(settings);
      if (response.data.status) {
        setMessage({ type: "success", text: "تم حفظ الإعدادات بنجاح" });
      }
    } catch (error) {
      setMessage({
        type: "error",
        text: handleAPIError(error) || "فشل حفظ الإعدادات",
      });
    } finally {
      setSaving(false);
    }
  };

  return (
    <div className="space-y-6">
      {loading ? (
        <div className="flex items-center justify-center py-12">
          <div className="text-center">
            <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 mx-auto"></div>
            <p className="text-gray-600 mt-4">جاري التحميل...</p>
          </div>
        </div>
      ) : (
        <>
          {/* Header */}
          <div className="flex items-center justify-between">
            <div>
              <h1 className="text-2xl font-bold text-gray-900">
                إعدادات النظام
              </h1>
              <p className="text-gray-600 mt-1">
                إدارة إعدادات وتكوينات النظام
              </p>
            </div>
            <Button onClick={handleSave} disabled={saving}>
              <Save className="w-4 h-4 ml-2" />
              {saving ? "جاري الحفظ..." : "حفظ التغييرات"}
            </Button>
          </div>

          {/* Message */}
          {message && (
            <div
              className={`p-4 rounded-lg ${
                message.type === "success"
                  ? "bg-green-50 text-green-800 border border-green-200"
                  : "bg-red-50 text-red-800 border border-red-200"
              }`}
            >
              {message.text}
            </div>
          )}

          {/* General Settings */}
          <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
            <div className="flex items-center gap-3 mb-6">
              <Globe className="w-5 h-5 text-blue-600" />
              <h2 className="text-lg font-semibold text-gray-900">
                الإعدادات العامة
              </h2>
            </div>

            <div className="space-y-4">
              {/* <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  اسم الموقع
                </label>
                <input
                  type="text"
                  value={settings.site_name}
                  onChange={(e) => handleChange("site_name", e.target.value)}
                  className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                />
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  البريد الإلكتروني
                </label>
                <input
                  type="email"
                  value={settings.site_email}
                  onChange={(e) => handleChange("site_email", e.target.value)}
                  className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                />
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  رابط الموقع
                </label>
                <input
                  type="url"
                  value={settings.site_url}
                  onChange={(e) => handleChange("site_url", e.target.value)}
                  className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  dir="ltr"
                />
              </div> */}

              <div className="flex items-center gap-3">
                <input
                  type="checkbox"
                  id="registration_open"
                  checked={settings.registration_open}
                  onChange={(e) =>
                    handleChange("registration_open", e.target.checked)
                  }
                  className="w-4 h-4 text-blue-600 border-gray-300 rounded focus:ring-blue-500"
                />
                <label
                  htmlFor="registration_open"
                  className="text-sm font-medium text-gray-700"
                >
                  السماح بالتسجيل
                </label>
              </div>

              <div className="flex items-center gap-3">
                <input
                  type="checkbox"
                  id="maintenance_mode"
                  checked={settings.maintenance_mode}
                  onChange={(e) =>
                    handleChange("maintenance_mode", e.target.checked)
                  }
                  className="w-4 h-4 text-blue-600 border-gray-300 rounded focus:ring-blue-500"
                />
                <label
                  htmlFor="maintenance_mode"
                  className="text-sm font-medium text-gray-700"
                >
                  وضع الصيانة
                </label>
              </div>
            </div>
          </div>

          {/* Exam Settings */}
          <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
            <div className="flex items-center gap-3 mb-6">
              <FileText className="w-5 h-5 text-blue-600" />
              <h2 className="text-lg font-semibold text-gray-900">
                إعدادات الاختبارات
              </h2>
            </div>

            <div className="space-y-4">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  مدة الاختبار (دقائق)
                </label>
                <input
                  type="number"
                  value={settings.exam_duration}
                  onChange={(e) =>
                    handleChange("exam_duration", parseInt(e.target.value))
                  }
                  className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  min="1"
                />
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  درجة النجاح (%)
                </label>
                <input
                  type="number"
                  value={settings.passing_score}
                  onChange={(e) =>
                    handleChange("passing_score", parseInt(e.target.value))
                  }
                  className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  min="0"
                  max="100"
                />
              </div>

              <div className="flex items-center gap-3">
                <input
                  type="checkbox"
                  id="auto_publish_results"
                  checked={settings.auto_publish_results}
                  onChange={(e) =>
                    handleChange("auto_publish_results", e.target.checked)
                  }
                  className="w-4 h-4 text-blue-600 border-gray-300 rounded focus:ring-blue-500"
                />
                <label
                  htmlFor="auto_publish_results"
                  className="text-sm font-medium text-gray-700"
                >
                  نشر النتائج تلقائياً
                </label>
              </div>
            </div>
          </div>

          {/* Security Settings */}
          <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
            <div className="flex items-center gap-3 mb-6">
              <Shield className="w-5 h-5 text-blue-600" />
              <h2 className="text-lg font-semibold text-gray-900">
                إعدادات الأمان
              </h2>
            </div>

            <div className="space-y-4">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  الحد الأقصى لمحاولات تسجيل الدخول
                </label>
                <input
                  type="number"
                  value={settings.max_login_attempts}
                  onChange={(e) =>
                    handleChange("max_login_attempts", parseInt(e.target.value))
                  }
                  className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  min="1"
                />
              </div>
            </div>
          </div>

          {/* Notification Settings */}
          {/* <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
            <div className="flex items-center gap-3 mb-6">
              <Bell className="w-5 h-5 text-blue-600" />
              <h2 className="text-lg font-semibold text-gray-900">
                إعدادات الإشعارات
              </h2>
            </div>

            <div className="space-y-4">
              <div className="flex items-center gap-3">
                <input
                  type="checkbox"
                  id="email_notifications"
                  checked={settings.email_notifications}
                  onChange={(e) =>
                    handleChange("email_notifications", e.target.checked)
                  }
                  className="w-4 h-4 text-blue-600 border-gray-300 rounded focus:ring-blue-500"
                />
                <label
                  htmlFor="email_notifications"
                  className="text-sm font-medium text-gray-700"
                >
                  إشعارات البريد الإلكتروني
                </label>
              </div>

              <div className="flex items-center gap-3">
                <input
                  type="checkbox"
                  id="sms_notifications"
                  checked={settings.sms_notifications}
                  onChange={(e) =>
                    handleChange("sms_notifications", e.target.checked)
                  }
                  className="w-4 h-4 text-blue-600 border-gray-300 rounded focus:ring-blue-500"
                />
                <label
                  htmlFor="sms_notifications"
                  className="text-sm font-medium text-gray-700"
                >
                  إشعارات الرسائل النصية
                </label>
              </div>
            </div>
          </div> */}

          {/* System Info */}
          <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
            <div className="flex items-center gap-3 mb-6">
              <Database className="w-5 h-5 text-blue-600" />
              <h2 className="text-lg font-semibold text-gray-900">
                معلومات النظام
              </h2>
            </div>

            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div className="p-4 bg-gray-50 rounded-lg">
                <p className="text-sm text-gray-600">إصدار النظام</p>
                <p className="text-lg font-semibold text-gray-900 mt-1">
                  1.1.3
                </p>
              </div>

              <div className="p-4 bg-gray-50 rounded-lg">
                <p className="text-sm text-gray-600">قاعدة البيانات</p>
                <p className="text-lg font-semibold text-gray-900 mt-1">
                  MySQL
                </p>
              </div>

              <div className="p-4 bg-gray-50 rounded-lg">
                <p className="text-sm text-gray-600">PHP</p>
                <p className="text-lg font-semibold text-gray-900 mt-1">8.3</p>
              </div>

              <div className="p-4 bg-gray-50 rounded-lg">
                <p className="text-sm text-gray-600">Laravel</p>
                <p className="text-lg font-semibold text-gray-900 mt-1">12.x</p>
              </div>
            </div>
          </div>
        </>
      )}
    </div>
  );
}
