"use client";

import { useEffect, useState } from "react";
import { useRouter } from "next/navigation";
import { userAPI, handleAPIError } from "@/lib/api";
import {
  FileText,
  Clock,
  AlertCircle,
  CheckCircle,
  TrendingUp,
  Award,
} from "lucide-react";
import Link from "next/link";

export default function ExamsListPage() {
  const router = useRouter();
  const [exams, setExams] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState("");

  useEffect(() => {
    loadExams();
  }, []);

  const loadExams = async () => {
    try {
      const response = await userAPI.getExams();
      setExams(response.data.data || []);
    } catch (err: any) {
      setError(handleAPIError(err));
    } finally {
      setLoading(false);
    }
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center min-h-screen">
        <div className="text-center">
          <div className="w-12 h-12 border-4 border-blue-600 border-t-transparent rounded-full animate-spin mx-auto" />
          <p className="text-gray-600 mt-4">جاري التحميل...</p>
        </div>
      </div>
    );
  }

  return (
    <div className="space-y-6" dir="rtl">
      {/* Header */}
      <div className="bg-white rounded-xl shadow-lg p-6">
        <div className="flex items-center gap-3">
          <div className="w-12 h-12 bg-blue-100 rounded-lg flex items-center justify-center">
            <FileText className="w-6 h-6 text-blue-600" />
          </div>
          <div>
            <h1 className="text-2xl font-bold text-gray-900">الاختبارات</h1>
            <p className="text-gray-600">قائمة الاختبارات المتاحة</p>
          </div>
        </div>
      </div>

      {/* Error Message */}
      {error && (
        <div className="bg-red-50 border border-red-200 rounded-lg p-4 flex items-start gap-3">
          <AlertCircle className="w-5 h-5 text-red-600 flex-shrink-0 mt-0.5" />
          <p className="text-sm text-red-800">{error}</p>
        </div>
      )}

      {/* Exams List */}
      {exams.length > 0 ? (
        <div className="grid gap-4">
          {exams.map((exam) => (
            <div
              key={exam.id}
              className="bg-white rounded-xl shadow-lg p-6 hover:shadow-xl transition-shadow"
            >
              <div className="flex items-start justify-between gap-4">
                <div className="flex-1">
                  <h3 className="text-xl font-semibold text-gray-900 mb-2">
                    {exam.title}
                  </h3>
                  {exam.description && (
                    <p className="text-gray-600 mb-4">{exam.description}</p>
                  )}

                  <div className="flex flex-wrap items-center gap-4 text-sm">
                    <div className="flex items-center gap-2 text-gray-600">
                      <Clock className="w-4 h-4" />
                      <span>المدة: {exam.duration_minutes} دقيقة</span>
                    </div>

                    {exam.deadline_at && (
                      <div className="flex items-center gap-2 text-gray-600">
                        <AlertCircle className="w-4 h-4" />
                        <span>
                          الموعد النهائي:{" "}
                          {new Date(exam.deadline_at).toLocaleDateString(
                            "ar-EG"
                          )}
                        </span>
                      </div>
                    )}

                    {exam.is_completed && (
                      <div className="flex items-center gap-2 text-green-600">
                        <CheckCircle className="w-4 h-4" />
                        <span className="font-medium">تم إكمال الاختبار</span>
                      </div>
                    )}
                  </div>

                  {/* Show results if available */}
                  {exam.results && (
                    <div className="mt-4 pt-4 border-t border-gray-200">
                      <div className="flex items-center gap-4">
                        <div
                          className={`flex items-center gap-2 px-4 py-2 rounded-lg ${
                            exam.results.passed
                              ? "bg-green-50 text-green-700"
                              : "bg-red-50 text-red-700"
                          }`}
                        >
                          <TrendingUp className="w-4 h-4" />
                          <span className="font-bold text-lg">
                            {Math.round(parseFloat(exam.results.score))}%
                          </span>
                        </div>

                        <div
                          className={`flex items-center gap-2 px-4 py-2 rounded-lg ${
                            exam.results.passed
                              ? "bg-green-100 text-green-800"
                              : "bg-red-100 text-red-800"
                          }`}
                        >
                          <Award className="w-4 h-4" />
                          <span className="font-medium">
                            {exam.results.passed ? "ناجح" : "راسب"}
                          </span>
                        </div>

                        <span className="text-sm text-gray-500">
                          الحد الأدنى: {exam.results.passing_score}%
                        </span>
                      </div>
                    </div>
                  )}
                </div>

                <Link
                  href={`/dashboard/exam/${exam.id}`}
                  className={`px-6 py-3 rounded-lg font-medium transition-colors ${
                    exam.is_completed
                      ? exam.results
                        ? "bg-blue-100 text-blue-600 hover:bg-blue-200"
                        : "bg-gray-100 text-gray-600"
                      : "bg-blue-600 hover:bg-blue-700 text-white"
                  }`}
                >
                  {exam.is_completed
                    ? exam.results
                      ? "عرض النتيجة"
                      : "مكتمل"
                    : "بدء الاختبار"}
                </Link>
              </div>
            </div>
          ))}
        </div>
      ) : (
        <div className="bg-white rounded-xl shadow-lg p-12 text-center">
          <FileText className="w-16 h-16 text-gray-400 mx-auto mb-4" />
          <h3 className="text-xl font-semibold text-gray-900 mb-2">
            لا توجد اختبارات متاحة
          </h3>
          <p className="text-gray-500">
            لا توجد اختبارات منشورة حالياً. سيتم إشعارك عند نشر اختبار جديد.
          </p>
          <Link
            href="/dashboard"
            className="inline-block mt-6 px-6 py-3 bg-blue-600 hover:bg-blue-700 text-white rounded-lg font-medium transition-colors"
          >
            العودة للوحة التحكم
          </Link>
        </div>
      )}
    </div>
  );
}
