"use client";

import { useEffect, useState } from "react";
import { judgeAPI, handleAPIError } from "@/lib/api";
import { Plus, Edit2, Trash2, FileText } from "lucide-react";
import { Button } from "@/components/ui/Button";

interface EvaluationSheet {
  id: number;
  title: string;
  criteria: Array<{
    name: string;
    max_score: number;
  }>;
  created_at: string;
}

export default function EvaluationSheetsPage() {
  const [sheets, setSheets] = useState<EvaluationSheet[]>([]);
  const [loading, setLoading] = useState(true);
  const [showModal, setShowModal] = useState(false);
  const [formData, setFormData] = useState({
    title: "",
    criteria: [{ name: "", max_score: 10 }],
  });

  useEffect(() => {
    loadSheets();
  }, []);

  const loadSheets = async () => {
    try {
      const response = await judgeAPI.getEvaluationSheets();
      setSheets(response.data.data);
    } catch (err: any) {
      console.error(handleAPIError(err));
    } finally {
      setLoading(false);
    }
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center h-screen">
        <div className="w-12 h-12 border-4 border-blue-600 border-t-transparent rounded-full animate-spin" />
      </div>
    );
  }

  return (
    <div className="space-y-6" dir="rtl">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold text-gray-900">نماذج التقييم</h1>
          <p className="text-gray-600 mt-1">إدارة نماذج ومعايير التقييم</p>
        </div>
      </div>

      {/* Sheets List */}
      {sheets.length === 0 ? (
        <div className="bg-white rounded-xl shadow-lg p-12 text-center">
          <FileText className="w-16 h-16 text-gray-400 mx-auto mb-4" />
          <p className="text-gray-500 mb-4">لا توجد نماذج تقييم بعد</p>
        </div>
      ) : (
        <div className="space-y-4">
          {sheets.map((sheet) => (
            <div key={sheet.id} className="bg-white rounded-xl shadow-lg p-6">
              <div className="flex items-start justify-between mb-4">
                <div className="flex items-center gap-3">
                  <div className="p-3 bg-blue-50 rounded-lg">
                    <FileText className="w-6 h-6 text-blue-600" />
                  </div>
                  <div>
                    <h3 className="text-lg font-semibold text-gray-900">
                      {sheet.title}
                    </h3>
                    <p className="text-sm text-gray-500">
                      {new Date(sheet.created_at).toLocaleDateString("ar-EG")}
                    </p>
                  </div>
                </div>
              </div>

              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-3">
                {sheet.criteria.map((criterion, index) => (
                  <div
                    key={index}
                    className="flex items-center justify-between p-3 bg-gray-50 rounded-lg"
                  >
                    <span className="text-sm font-medium text-gray-900">
                      {criterion.name}
                    </span>
                    <span className="text-sm text-gray-600">
                      الحد الأقصى: {criterion.max_score}
                    </span>
                  </div>
                ))}
              </div>
            </div>
          ))}
        </div>
      )}
    </div>
  );
}
