"use client";

import { useEffect, useState } from "react";
import { useRouter } from "next/navigation";
import { judgeAPI, handleAPIError } from "@/lib/api";
import {
  BarChart3,
  AlertCircle,
  Search,
  ChevronRight,
  CheckCircle,
  Clock,
  TrendingUp,
} from "lucide-react";

interface Evaluation {
  id: number;
  user_id: number;
  user_name: string;
  user_email: string;
  evaluation_sheet_title: string;
  score: number | string;
  max_score: number;
  submitted_at: string;
}

interface Statistics {
  total_assigned_users?: number;
  pending_evaluations?: number;
  completed_evaluations?: number;
  avg_score_given?: number;
  evaluation_sheets_count?: number;
  total_evaluations?: number;
  total_sheets?: number;
  evaluations_this_month?: number;
}

export default function MyEvaluationsPage() {
  const router = useRouter();
  const [evaluations, setEvaluations] = useState<Evaluation[]>([]);
  const [loading, setLoading] = useState(true);
  const [search, setSearch] = useState("");
  const [stats, setStats] = useState<Statistics | null>(null);

  useEffect(() => {
    loadData();
  }, []);

  // Single API call to get evaluations which contains the data we need
  const loadData = async () => {
    try {
      const response = await judgeAPI.getMyEvaluations();
      const evaluationsData = Array.isArray(response.data.data)
        ? response.data.data
        : response.data.data?.data || [];

      setEvaluations(evaluationsData);

      // Calculate stats from evaluations data instead of making another API call
      if (evaluationsData.length > 0) {
        const totalScore = evaluationsData.reduce(
          (sum: number, evaluation: Evaluation) =>
            sum +
            (typeof evaluation.score === "string"
              ? parseFloat(evaluation.score)
              : evaluation.score),
          0
        );
        const avgScore = totalScore / evaluationsData.length;

        setStats({
          total_evaluations: evaluationsData.length,
          completed_evaluations: evaluationsData.length,
          avg_score_given: Math.round(avgScore * 100) / 100,
          evaluation_sheets_count: new Set(
            evaluationsData.map(
              (evaluation: Evaluation) => evaluation.evaluation_sheet_title
            )
          ).size,
          total_sheets: new Set(
            evaluationsData.map(
              (evaluation: Evaluation) => evaluation.evaluation_sheet_title
            )
          ).size,
          evaluations_this_month: evaluationsData.filter(
            (evaluation: Evaluation) => {
              const evalDate = new Date(evaluation.submitted_at);
              const now = new Date();
              return (
                evalDate.getMonth() === now.getMonth() &&
                evalDate.getFullYear() === now.getFullYear()
              );
            }
          ).length,
        });
      } else {
        setStats({
          total_evaluations: 0,
          completed_evaluations: 0,
          avg_score_given: 0,
          evaluation_sheets_count: 0,
          total_sheets: 0,
          evaluations_this_month: 0,
        });
      }
    } catch (err: any) {
      console.error(handleAPIError(err));
    } finally {
      setLoading(false);
    }
  };

  const filteredEvaluations = evaluations.filter(
    (evaluation) =>
      evaluation.user_name.toLowerCase().includes(search.toLowerCase()) ||
      evaluation.user_email.toLowerCase().includes(search.toLowerCase())
  );

  if (loading) {
    return (
      <div className="flex items-center justify-center h-screen">
        <div className="w-12 h-12 border-4 border-blue-600 border-t-transparent rounded-full animate-spin" />
      </div>
    );
  }

  return (
    <div className="space-y-6" dir="rtl">
      {/* Header */}
      <div>
        <h1 className="text-3xl font-bold text-gray-900">تقييماتي</h1>
        <p className="text-gray-600 mt-1">عرض جميع التقييمات التي قمت بها</p>
      </div>

      {/* Statistics Cards */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
        <div className="bg-gradient-to-br from-blue-500 to-blue-600 rounded-xl shadow-lg p-6 text-white">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-blue-100 mb-1">إجمالي التقييمات</p>
              <p className="text-3xl font-bold">
                {stats?.total_evaluations || 0}
              </p>
            </div>
            <BarChart3 className="w-10 h-10 text-blue-100 opacity-50" />
          </div>
        </div>

        <div className="bg-gradient-to-br from-green-500 to-green-600 rounded-xl shadow-lg p-6 text-white">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-green-100 mb-1">متوسط الدرجات</p>
              <p className="text-3xl font-bold">
                {(stats?.avg_score_given || 0).toFixed(1)}
              </p>
            </div>
            <TrendingUp className="w-10 h-10 text-green-100 opacity-50" />
          </div>
        </div>

        <div className="bg-gradient-to-br from-purple-500 to-purple-600 rounded-xl shadow-lg p-6 text-white">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-purple-100 mb-1">المكتملة</p>
              <p className="text-3xl font-bold">
                {stats?.completed_evaluations || 0}
              </p>
            </div>
            <CheckCircle className="w-10 h-10 text-purple-100 opacity-50" />
          </div>
        </div>

        <div className="bg-gradient-to-br from-orange-500 to-orange-600 rounded-xl shadow-lg p-6 text-white">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-orange-100 mb-1">هذا الشهر</p>
              <p className="text-3xl font-bold">
                {stats?.evaluations_this_month || 0}
              </p>
            </div>
            <Clock className="w-10 h-10 text-orange-100 opacity-50" />
          </div>
        </div>
      </div>

      {/* Search */}
      <div className="bg-white rounded-xl shadow-lg p-6">
        <div className="relative">
          <Search className="absolute right-3 top-1/2 transform -translate-y-1/2 w-5 h-5 text-gray-400" />
          <input
            type="text"
            value={search}
            onChange={(e) => setSearch(e.target.value)}
            placeholder="البحث بالاسم أو البريد الإلكتروني..."
            className="w-full pr-11 pl-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
          />
        </div>
      </div>

      {/* Evaluations List */}
      {filteredEvaluations.length === 0 ? (
        <div className="bg-white rounded-xl shadow-lg p-12 text-center">
          <AlertCircle className="w-16 h-16 text-gray-400 mx-auto mb-4" />
          <p className="text-gray-500">
            {evaluations.length === 0
              ? "لم تقم بأي تقييمات بعد"
              : "لم يتم العثور على تقييمات مطابقة"}
          </p>
        </div>
      ) : (
        <div className="space-y-4">
          {filteredEvaluations.map((evaluation) => (
            <div
              key={evaluation.id}
              className="bg-white rounded-xl shadow-md hover:shadow-xl transition-all duration-300 overflow-hidden border border-gray-100 cursor-pointer group"
              onClick={() => router.push(`/judge/users/${evaluation.user_id}`)}
            >
              <div className="p-6">
                <div className="flex items-start justify-between mb-4">
                  <div className="flex items-center gap-4 flex-1">
                    <div className="w-12 h-12 bg-gradient-to-br from-blue-500 to-indigo-600 rounded-full flex items-center justify-center shadow-md group-hover:shadow-lg transition-shadow">
                      <span className="text-white font-semibold text-lg">
                        {evaluation.user_name.charAt(0)}
                      </span>
                    </div>
                    <div className="flex-1">
                      <h3 className="font-semibold text-gray-900 text-lg group-hover:text-blue-600 transition-colors">
                        {evaluation.user_name}
                      </h3>
                      <p className="text-sm text-gray-500">
                        {evaluation.user_email}
                      </p>
                      <p className="text-xs text-gray-400 mt-1">
                        نموذج: {evaluation.evaluation_sheet_title}
                      </p>
                    </div>
                  </div>
                  <ChevronRight className="w-6 h-6 text-gray-300 group-hover:text-blue-600 transition-all group-hover:translate-x-1" />
                </div>

                <div className="grid grid-cols-3 gap-3">
                  <div className="bg-gradient-to-br from-blue-50 to-blue-100 rounded-lg p-3 border border-blue-200">
                    <p className="text-xs text-blue-600 font-semibold mb-1">
                      الدرجة
                    </p>
                    <p className="text-xl font-bold text-blue-700">
                      {evaluation.score}/{evaluation.max_score}
                    </p>
                  </div>
                  <div className="bg-gradient-to-br from-green-50 to-green-100 rounded-lg p-3 border border-green-200">
                    <p className="text-xs text-green-600 font-semibold mb-1">
                      النسبة %
                    </p>
                    <p className="text-xl font-bold text-green-700">
                      {(
                        ((typeof evaluation.score === "string"
                          ? parseFloat(evaluation.score)
                          : evaluation.score) /
                          evaluation.max_score) *
                        100
                      ).toFixed(0)}
                      %
                    </p>
                  </div>
                  <div className="bg-gradient-to-br from-purple-50 to-purple-100 rounded-lg p-3 border border-purple-200">
                    <p className="text-xs text-purple-600 font-semibold mb-1">
                      التاريخ
                    </p>
                    <p className="text-sm font-medium text-purple-700">
                      {new Date(evaluation.submitted_at).toLocaleDateString(
                        "ar-EG"
                      )}
                    </p>
                  </div>
                </div>
              </div>
            </div>
          ))}
        </div>
      )}
    </div>
  );
}
