"use client";

import { useEffect, useState } from "react";
import { judgeAPI, handleAPIError } from "@/lib/api";
import {
  Users,
  FileText,
  BarChart3,
  Award,
  TrendingUp,
  Clock,
  CheckCircle,
} from "lucide-react";

interface Statistics {
  total_assigned_users?: number;
  pending_evaluations?: number;
  completed_evaluations?: number;
  avg_score_given?: number;
  evaluation_sheets_count?: number;
  total_evaluations?: number;
  total_sheets?: number;
  evaluations_this_month?: number;
}

interface Evaluation {
  id: number;
  user_id: number;
  user_name: string;
  user_email: string;
  evaluation_sheet_title: string;
  score: number;
  max_score: number;
  submitted_at: string;
}

export default function JudgeDashboard() {
  const [stats, setStats] = useState<Statistics | null>(null);
  const [recentEvaluations, setRecentEvaluations] = useState<Evaluation[]>([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    loadStatistics();
    loadRecentEvaluations();
  }, []);

  const loadStatistics = async () => {
    try {
      const response = await judgeAPI.getStatistics();
      setStats(response.data.data);
    } catch (err: any) {
      console.error(handleAPIError(err));
    } finally {
      setLoading(false);
    }
  };

  const loadRecentEvaluations = async () => {
    try {
      const response = await judgeAPI.getMyEvaluations({ per_page: 5 });
      const evaluations = Array.isArray(response.data.data)
        ? response.data.data
        : response.data.data?.data || [];
      setRecentEvaluations(evaluations);
    } catch (err: any) {
      console.error(handleAPIError(err));
    }
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center h-screen">
        <div className="w-12 h-12 border-4 border-blue-600 border-t-transparent rounded-full animate-spin" />
      </div>
    );
  }

  const statCards = [
    {
      title: "إجمالي التقييمات",
      value: stats?.total_evaluations || 0,
      icon: Users,
      gradient: "from-blue-500 to-blue-600",
    },
    {
      title: "المكتملة",
      value: stats?.completed_evaluations || 0,
      icon: Award,
      gradient: "from-green-500 to-green-600",
    },
    {
      title: "قيد الانتظار",
      value: stats?.pending_evaluations || 0,
      icon: BarChart3,
      gradient: "from-yellow-500 to-yellow-600",
    },
    {
      title: "متوسط درجاتي",
      value: (stats?.avg_score_given || 0).toFixed(2),
      icon: Award,
      gradient: "from-purple-500 to-purple-600",
    },
    {
      title: "نماذج التقييم",
      value: stats?.total_sheets || 0,
      icon: FileText,
      gradient: "from-indigo-500 to-indigo-600",
    },
    {
      title: "هذا الشهر",
      value: stats?.evaluations_this_month || 0,
      icon: BarChart3,
      gradient: "from-orange-500 to-orange-600",
    },
  ];

  return (
    <div className="space-y-6" dir="rtl">
      <div>
        <h1 className="text-3xl font-bold text-gray-900">لوحة المقيّم</h1>
        <p className="text-gray-600 mt-1">إدارة التقييمات والمتقدمين</p>
      </div>

      {/* Statistics Cards */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-6 gap-4">
        {statCards.map((card) => {
          const Icon = card.icon;
          return (
            <div
              key={card.title}
              className="bg-white rounded-xl shadow-lg p-4 hover:shadow-xl transition-shadow"
            >
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-gray-600 mb-1">{card.title}</p>
                  <p className="text-2xl font-bold text-gray-900">
                    {card.value}
                  </p>
                </div>
                <div
                  className={`p-3 rounded-lg bg-gradient-to-br ${card.gradient}`}
                >
                  <Icon className="w-6 h-6 text-white" />
                </div>
              </div>
            </div>
          );
        })}
      </div>

      {/* Quick Actions */}
      <div className="bg-gradient-to-br from-blue-500 to-indigo-600 rounded-xl shadow-lg p-6 text-black">
        <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
          <TrendingUp className="w-6 h-6" />
          إجراءات سريعة
        </h3>
        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
          <a
            href="/judge/users"
            className="block px-6 py-4 bg-white bg-opacity-20 rounded-lg hover:bg-opacity-30 transition-colors text-center"
          >
            <Users className="w-8 h-8 mx-auto mb-2" />
            <span className="font-medium">تقييم المتقدمين</span>
          </a>
          <a
            href="/judge/evaluation-sheets"
            className="block px-6 py-4 bg-white bg-opacity-20 rounded-lg hover:bg-opacity-30 transition-colors text-center"
          >
            <FileText className="w-8 h-8 mx-auto mb-2" />
            <span className="font-medium">نماذج التقييم</span>
          </a>
          <a
            href="/judge/evaluations"
            className="block px-6 py-4 bg-white bg-opacity-20 rounded-lg hover:bg-opacity-30 transition-colors text-center"
          >
            <BarChart3 className="w-8 h-8 mx-auto mb-2" />
            <span className="font-medium">تقييماتي</span>
          </a>
        </div>
      </div>

      {/* Recent Activity */}
      <div className="bg-white rounded-xl shadow-lg p-6">
        <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center gap-2">
          <Clock className="w-5 h-5 text-blue-600" />
          نشاط حديث
        </h3>
        {recentEvaluations.length > 0 ? (
          <div className="space-y-3">
            {recentEvaluations.map((evaluation) => (
              <div
                key={evaluation.id}
                className="flex items-start gap-4 p-4 border border-gray-200 rounded-lg hover:bg-gray-50 transition-colors"
              >
                <div className="flex-shrink-0">
                  <div className="flex items-center justify-center h-10 w-10 rounded-full bg-green-100">
                    <CheckCircle className="h-6 w-6 text-green-600" />
                  </div>
                </div>
                <div className="flex-1 min-w-0">
                  <p className="text-sm font-medium text-gray-900">
                    تقييم {evaluation.user_name}
                  </p>
                  <p className="text-sm text-gray-600 mt-1">
                    نموذج: {evaluation.evaluation_sheet_title}
                  </p>
                  <p className="text-sm text-gray-500 mt-1">
                    الدرجة: {evaluation.score} / {evaluation.max_score}
                  </p>
                  <p className="text-xs text-gray-400 mt-2">
                    {new Date(evaluation.submitted_at).toLocaleString("ar-EG")}
                  </p>
                </div>
              </div>
            ))}
          </div>
        ) : (
          <div className="text-center py-8">
            <Clock className="w-12 h-12 text-gray-300 mx-auto mb-3" />
            <p className="text-gray-500">لا توجد أنشطة حديثة</p>
          </div>
        )}
      </div>
    </div>
  );
}
