import type { Metadata, Viewport } from "next";
import { Cairo } from "next/font/google";
import { Footer } from "@/components/shared/footer";
import "./globals.css";

// Cairo Font
const cairoFont = Cairo({
  subsets: ["arabic", "latin"],
  weight: ["300", "400", "500", "600", "700", "800", "900"],
  variable: "--font-cairo",
  display: "swap",
});

// ==========================
//        SEO METADATA
// ==========================
const DOMAIN = process.env.NEXT_PUBLIC_DOMAIN || "https://your-domain.com";
const SITE_NAME = "برنامج الدبلوماسية الشبابية";
const DESCRIPTION =
  "استمارة رسمية مقدمة من وزارة الشباب والرياضة للتقدم إلى برنامج الدبلوماسية الشبابية، الذي يهدف إلى تأهيل وإعداد شباب قادر على تمثيل مصر دوليًا";

export const metadata: Metadata = {
  metadataBase: new URL(DOMAIN),

  title: {
    default: `استمارة التقدم لبرنامج الدبلوماسية الشبابية | ${SITE_NAME}`,
    template: `%s | ${SITE_NAME}`,
  },

  description: DESCRIPTION,

  keywords: [
    "وزارة الشباب والرياضة",
    "برنامج الدبلوماسية الشبابية",
    "استمارة التقدم",
    "نموذج تسجيل",
    "التقديم في الدبلوماسية الشبابية",
    "برامج تدريب الشباب",
    "تمكين الشباب",
    "العمل الدبلوماسي",
    "الدبلوماسية المصرية",
    "برامج وزارة الشباب",
    "قيادة الشباب",
    "التنمية البشرية",
    "المهارات الدبلوماسية",
  ],

  creator: "وزارة الشباب والرياضة",
  publisher: "وزارة الشباب والرياضة",

  openGraph: {
    type: "website",
    locale: "ar_EG",
    url: DOMAIN,
    siteName: SITE_NAME,
    title: `استمارة التقدم لبرنامج الدبلوماسية الشبابية`,
    description: DESCRIPTION,
    images: [
      {
        url: "/og-image.png",
        width: 1200,
        height: 630,
        alt: "استمارة التقدم لبرنامج الدبلوماسية الشبابية",
        type: "image/png",
      },
    ],
  },

  twitter: {
    card: "summary_large_image",
    title: "استمارة التقدم لبرنامج الدبلوماسية الشبابية",
    description: DESCRIPTION,
    images: ["/og-image.png"],
    creator: "@YouthDiplomacy",
  },

  robots: {
    index: true,
    follow: true,
    nocache: false,
    googleBot: {
      index: true,
      follow: true,
      "max-image-preview": "large",
      "max-snippet": -1,
      "max-video-preview": -1,
    },
  },

  icons: {
    icon: [
      { url: "/favicon-16x16.png", sizes: "16x16", type: "image/png" },
      { url: "/favicon-32x32.png", sizes: "32x32", type: "image/png" },
      { url: "/favicon-96x96.png", sizes: "96x96", type: "image/png" },
      { url: "/android-icon-192x192.png", sizes: "192x192", type: "image/png" },
    ],
    apple: [
      { url: "/apple-icon-57x57.png", sizes: "57x57" },
      { url: "/apple-icon-60x60.png", sizes: "60x60" },
      { url: "/apple-icon-72x72.png", sizes: "72x72" },
      { url: "/apple-icon-76x76.png", sizes: "76x76" },
      { url: "/apple-icon-114x114.png", sizes: "114x114" },
      { url: "/apple-icon-120x120.png", sizes: "120x120" },
      { url: "/apple-icon-144x144.png", sizes: "144x144" },
      { url: "/apple-icon-152x152.png", sizes: "152x152" },
      { url: "/apple-icon-180x180.png", sizes: "180x180" },
    ],
    shortcut: "/favicon-32x32.png",
  },

  manifest: "/manifest.json",
  alternates: {
    canonical: DOMAIN,
  },
};

// Viewport configuration (moved from metadata)
export const viewport: Viewport = {
  themeColor: "#ffffff",
  width: "device-width",
  initialScale: 1,
  maximumScale: 5,
};

// ==========================
//        ROOT LAYOUT
// ==========================
export default function RootLayout({
  children,
}: {
  children: React.ReactNode;
}) {
  return (
    <html lang="ar" dir="rtl" className={cairoFont.variable}>
      <head>
        {/* Preconnect to external domains for performance */}
        <link rel="preconnect" href="https://fonts.googleapis.com" />
        <link
          rel="preconnect"
          href="https://fonts.gstatic.com"
          crossOrigin="anonymous"
        />

        {/* Schema.org structured data */}
        <script
          type="application/ld+json"
          dangerouslySetInnerHTML={{
            __html: JSON.stringify({
              "@context": "https://schema.org",
              "@type": "Organization",
              name: SITE_NAME,
              url: DOMAIN,
              logo: `${DOMAIN}/android-icon-192x192.png`,
              description: DESCRIPTION,
              sameAs: [
                "https://www.facebook.com/YouthDiplomacy",
                "https://twitter.com/YouthDiplomacy",
              ],
              contactPoint: {
                "@type": "ContactPoint",
                contactType: "Customer Support",
                availableLanguage: ["ar", "en"],
              },
            }),
          }}
        />

        {/* Breadcrumb schema for homepage */}
        <script
          type="application/ld+json"
          dangerouslySetInnerHTML={{
            __html: JSON.stringify({
              "@context": "https://schema.org",
              "@type": "BreadcrumbList",
              itemListElement: [
                {
                  "@type": "ListItem",
                  position: 1,
                  name: "الرئيسية",
                  item: DOMAIN,
                },
              ],
            }),
          }}
        />
      </head>
      <body className="antialiased">{children}</body>
    </html>
  );
}
