"use client";

import { useEffect, useState } from "react";
import { adminAPI, handleAPIError } from "@/lib/api";
import { X, Check } from "lucide-react";

interface Category {
  id: number;
  name: string;
}

interface Props {
  isOpen: boolean;
  onClose: () => void;
  onSuccess: () => void;
  userId: number;
  userName: string;
  currentCategories: Array<{ id: number; name: string }>;
}

export default function AssignCategoriesModal({
  isOpen,
  onClose,
  onSuccess,
  userId,
  userName,
  currentCategories,
}: Props) {
  const [categories, setCategories] = useState<Category[]>([]);
  const [selectedIds, setSelectedIds] = useState<Set<number>>(new Set());
  const [loading, setLoading] = useState(true);
  const [assigning, setAssigning] = useState(false);

  useEffect(() => {
    if (isOpen) {
      loadCategories();
    }
  }, [isOpen]);

  const loadCategories = async () => {
    try {
      const response = await adminAPI.getCategories();
      const cats = response.data.data;
      setCategories(cats);

      // Initialize selected categories with current user categories
      const currentIds = new Set(currentCategories.map((c) => c.id));
      setSelectedIds(currentIds);
    } catch (err: any) {
      console.error(handleAPIError(err));
    } finally {
      setLoading(false);
    }
  };

  const toggleCategory = (categoryId: number) => {
    const newSelected = new Set(selectedIds);
    if (newSelected.has(categoryId)) {
      newSelected.delete(categoryId);
    } else {
      newSelected.add(categoryId);
    }
    setSelectedIds(newSelected);
  };

  const handleAssign = async () => {
    if (selectedIds.size === 0) {
      alert("يرجى تحديد واحد أو أكثر من التصنيفات");
      return;
    }

    setAssigning(true);
    try {
      // Get categories to add (not in current but in selected)
      const currentIds = new Set(currentCategories.map((c) => c.id));
      const toAdd = Array.from(selectedIds).filter((id) => !currentIds.has(id));
      const toRemove = Array.from(currentIds).filter(
        (id) => !selectedIds.has(id)
      );

      if (toAdd.length > 0) {
        await adminAPI.assignCategoriesToUser(userId, toAdd);
      }

      if (toRemove.length > 0) {
        await adminAPI.removeCategoriesFromUser(userId, toRemove);
      }

      onSuccess();
      onClose();
    } catch (err: any) {
      alert(handleAPIError(err));
    } finally {
      setAssigning(false);
    }
  };

  if (!isOpen) return null;

  return (
    <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
      <div className="bg-white rounded-xl max-w-md w-full p-6" dir="rtl">
        <div className="flex items-center justify-between mb-4">
          <h3 className="text-xl font-bold text-gray-900">تعيين التصنيفات</h3>
          <button
            onClick={onClose}
            className="p-1 hover:bg-gray-100 rounded-lg transition-colors"
          >
            <X className="w-5 h-5 text-gray-600" />
          </button>
        </div>

        <p className="text-sm text-gray-600 mb-4">
          المستخدم: <span className="font-semibold">{userName}</span>
        </p>

        {loading ? (
          <div className="flex items-center justify-center p-8">
            <div className="w-8 h-8 border-4 border-blue-600 border-t-transparent rounded-full animate-spin" />
          </div>
        ) : categories.length === 0 ? (
          <div className="text-center p-8">
            <p className="text-gray-500">لا توجد تصنيفات متاحة</p>
          </div>
        ) : (
          <>
            <div className="space-y-2 max-h-64 overflow-y-auto mb-6">
              {categories.map((category) => (
                <label
                  key={category.id}
                  className="flex items-center gap-3 p-3 hover:bg-gray-50 rounded-lg cursor-pointer transition-colors"
                >
                  <input
                    type="checkbox"
                    checked={selectedIds.has(category.id)}
                    onChange={() => toggleCategory(category.id)}
                    className="w-4 h-4 border-gray-300 rounded cursor-pointer text-blue-600 focus:ring-blue-500"
                  />
                  <span className="text-gray-700">{category.name}</span>
                </label>
              ))}
            </div>

            <div className="flex gap-3">
              <button
                onClick={handleAssign}
                disabled={assigning}
                className="flex-1 flex items-center justify-center gap-2 bg-blue-600 hover:bg-blue-700 disabled:opacity-50 text-white py-3 rounded-lg transition-colors"
              >
                <Check className="w-4 h-4" />
                {assigning ? "جاري التحديث..." : "تحديث التصنيفات"}
              </button>
              <button
                onClick={onClose}
                disabled={assigning}
                className="flex-1 bg-gray-200 hover:bg-gray-300 text-gray-700 py-3 rounded-lg transition-colors"
              >
                إلغاء
              </button>
            </div>
          </>
        )}
      </div>
    </div>
  );
}
