"use client";

import React, { useState } from "react";
import { Upload, X, FileIcon } from "lucide-react";

interface FileUploadProps {
  label: string;
  name: string;
  required?: boolean;
  multiple?: boolean;
  accept?: string;
  register: any;
  setValue: any;
  error?: string;
}

export function FileUpload({
  label,
  name,
  required,
  multiple,
  accept,
  register,
  setValue,
  error,
}: FileUploadProps) {
  const [files, setFiles] = useState<File[]>([]);

  const handleChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const fileList = e.target.files;
    if (fileList) {
      // Convert FileList to array and store
      const fileArray = Array.from(fileList);
      setFiles(fileArray);
      // Set the actual File objects in the form, not the FileList
      setValue(name, fileArray, { shouldValidate: true });
    }
  };

  const handleRemove = () => {
    setFiles([]);
    // Clear the value from form state
    setValue(name, [], { shouldValidate: true });
    // Reset the file input
    const fileInput = document.getElementById(name) as HTMLInputElement;
    if (fileInput) {
      fileInput.value = "";
    }
  };

  // Register the field
  const registration = register(name);

  return (
    <div className="w-full">
      <label className="block text-sm font-medium text-gray-700 mb-2">
        {label}
        {required && <span className="text-red-500 ml-1">*</span>}
      </label>

      <div
        className={`border-2 border-dashed rounded-lg p-6 text-center transition-colors ${
          error ? "border-red-500" : "border-gray-300 hover:border-blue-500"
        }`}
      >
        <input
          type="file"
          id={name}
          multiple={multiple}
          accept={accept}
          onChange={(e) => {
            registration.onChange(e);
            handleChange(e);
          }}
          onBlur={registration.onBlur}
          name={registration.name}
          ref={registration.ref}
          className="hidden"
        />
        <label
          htmlFor={name}
          className="cursor-pointer flex flex-col items-center gap-2"
        >
          <Upload className="w-12 h-12 text-gray-400" />
          <span className="text-sm text-gray-600">
            انقر هنا لتحميل الملف{multiple ? "ات" : ""} أو اسحبها وأفلتها
          </span>
          {accept && (
            <span className="text-xs text-gray-500">
              الصيغ المقبولة: {accept}
            </span>
          )}
          {multiple && (
            <span className="text-xs text-gray-500">
              يُسمح بتحميل ملفات متعددة
            </span>
          )}
        </label>
      </div>

      {files.length > 0 && (
        <div className="mt-3 space-y-2">
          {files.map((file, idx) => (
            <div
              key={idx}
              className="flex items-center justify-between bg-blue-50 px-4 py-2 rounded"
            >
              <div className="flex items-center gap-2">
                <FileIcon className="w-4 h-4 text-blue-600" />
                <span className="text-sm text-gray-700">{file.name}</span>
              </div>
              {idx === 0 && (
                <button
                  type="button"
                  onClick={handleRemove}
                  className="text-red-500 hover:text-red-700"
                >
                  <X className="w-4 h-4" />
                </button>
              )}
            </div>
          ))}
        </div>
      )}

      {error && <p className="mt-1 text-sm text-red-500">{error}</p>}
    </div>
  );
}
