"use client";

import React, { useState, useEffect } from "react";
import { FormField as FormFieldType, FieldOption } from "@/lib/types";
import { Input } from "@/components/ui/Input";
import { Select } from "@/components/ui/Select";
import { FileUpload } from "@/components/form/FileUpload";
import { ArrayFieldInput } from "@/components/form/ArrayFieldInput";
import { DocumentsUpload } from "@/components/form/DocumentsUpload";
import { apiCall } from "@/lib/utils";

// Global persistent cache for options
const optionsCache = new Map<
  string,
  { data: any[]; fetching: Promise<any[]> | null }
>();

interface FormFieldProps {
  field: FormFieldType;
  register: any;
  errors: any;
  watch: any;
  setValue: any;
  control: any;
}

export function FormField({
  field,
  register,
  errors,
  watch,
  setValue,
  control,
}: FormFieldProps) {
  const [dynamicOptions, setDynamicOptions] = useState<any[]>([]);
  const [isLoadingOptions, setIsLoadingOptions] = useState(false);

  // Fetch dynamic options from endpoint - only once with caching
  useEffect(() => {
    if (!field.optionsEndpoint) return;

    // Check cache first - if already cached, use it immediately
    if (optionsCache.has(field.optionsEndpoint)) {
      const cached = optionsCache.get(field.optionsEndpoint)!;
      setDynamicOptions(cached.data);

      // If currently fetching, wait for it
      if (cached.fetching) {
        cached.fetching.then((options) => {
          setDynamicOptions(options);
        });
      }
      return;
    }

    // Not in cache, start fetching
    setIsLoadingOptions(true);

    // Create fetch promise
    const fetchPromise = apiCall(field.optionsEndpoint)
      .then((res) => res.json())
      .then((data) => {
        const options = Array.isArray(data)
          ? data
          : data.data || data.options || [];
        setDynamicOptions(options);
        // Update cache with final data
        if (field.optionsEndpoint) {
          const cacheEntry = optionsCache.get(field.optionsEndpoint);
          if (cacheEntry) {
            cacheEntry.data = options;
            cacheEntry.fetching = null;
          }
        }
        return options;
      })
      .catch((error) => {
        console.error(`Error fetching options for ${field.name}:`, error);
        setDynamicOptions([]);
        return [];
      })
      .finally(() => setIsLoadingOptions(false));

    // Store in cache with fetching promise
    optionsCache.set(field.optionsEndpoint, {
      data: [],
      fetching: fetchPromise,
    });
  }, [field.optionsEndpoint]);

  // Check conditional rendering
  if (field.conditional) {
    const watchValue = watch(field.conditional.field);
    let conditionMet = false;

    // Handle boolean conditions
    if (typeof field.conditional.value === "boolean") {
      // Convert string "true"/"false" to boolean if needed
      const normalizedWatchValue =
        watchValue === "true"
          ? true
          : watchValue === "false"
          ? false
          : watchValue;
      conditionMet = normalizedWatchValue === field.conditional.value;
    } else {
      // Handle string conditions
      conditionMet = String(watchValue) === String(field.conditional.value);
    }

    if (!conditionMet) {
      return null;
    }
  }

  // Get error message
  const error = errors[field.name]?.message;

  const renderField = () => {
    // Handle nested documents object
    if (field.name === "documents" && field.objectFields) {
      return (
        <DocumentsUpload
          field={field}
          register={register}
          errors={errors}
          setValue={setValue}
        />
      );
    }

    // Handle array fields (nationalities, degrees, languages)
    if (field.type === "array" && field.arrayFields) {
      return (
        <ArrayFieldInput
          field={field}
          control={control}
          errors={errors}
          watch={watch}
          register={register}
        />
      );
    }

    switch (field.type) {
      case "text":
      case "email":
      case "password":
      case "date":
      case "number":
      case "url":
        return (
          <Input
            type={field.type}
            label={field.label}
            placeholder={field.placeholder}
            required={field.required}
            readOnly={field.readonly}
            error={error}
            {...register(field.name)}
          />
        );

      case "select":
        // Use dynamic options if endpoint is provided, otherwise use static options
        const selectOptions = field.optionsEndpoint
          ? dynamicOptions
          : field.options || [];
        const optionStrings = selectOptions.map((opt) =>
          typeof opt === "string" ? opt : opt.label
        );
        const optionValues = selectOptions.map((opt) =>
          typeof opt === "string" ? opt : String(opt.value || opt.id)
        );

        return (
          <Select
            label={field.label}
            required={field.required}
            options={optionStrings}
            optionValues={optionValues}
            error={error}
            disabled={isLoadingOptions || selectOptions.length === 0}
            {...register(field.name)}
          />
        );

      case "radio":
        const radioOptions = field.options || [];
        return (
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-3">
              {field.label}
              {field.required && <span className="text-red-500 ml-1">*</span>}
            </label>
            <div className="space-y-2">
              {radioOptions.map((option) => {
                const isBooleanOption =
                  typeof option === "object" &&
                  "value" in option &&
                  typeof (option as any).value === "boolean";
                const optionValue = isBooleanOption
                  ? String((option as any).value)
                  : typeof option === "string"
                  ? option
                  : String((option as any).value);
                const optionLabel =
                  typeof option === "string" ? option : (option as any).label;

                return (
                  <label
                    key={optionValue}
                    className="flex items-center gap-2 cursor-pointer"
                  >
                    <input
                      type="radio"
                      value={optionValue}
                      {...register(field.name)}
                      className="w-4 h-4 text-blue-600"
                    />
                    <span className="text-gray-700">{optionLabel}</span>
                  </label>
                );
              })}
            </div>
            {error && <p className="mt-1 text-sm text-red-500">{error}</p>}
          </div>
        );

      case "textarea":
        return (
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              {field.label}
              {field.required && <span className="text-red-500 ml-1">*</span>}
            </label>
            <textarea
              rows={4}
              className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
              {...register(field.name)}
            />
            {error && <p className="mt-1 text-sm text-red-500">{error}</p>}
          </div>
        );

      case "file":
        return (
          <FileUpload
            label={field.label}
            name={field.name}
            required={field.required}
            multiple={field.multiple}
            accept={field.accept}
            register={register}
            setValue={setValue}
            error={error}
          />
        );

      default:
        return <Input label={field.label} {...register(field.name)} />;
    }
  };

  return <div className="mb-6">{renderField()}</div>;
}
