interface ProgressBarProps {
  currentSection: number;
  totalSections: number;
  sectionTitle?: string;
  sections?: Array<{ title: string }>;
  onStepClick?: (stepIndex: number) => void;
  validSections?: boolean[];
}

export function ProgressBar({
  currentSection,
  totalSections,
  sectionTitle,
  sections,
  onStepClick,
  validSections = [],
}: ProgressBarProps) {
  const progress = ((currentSection + 1) / totalSections) * 100;

  return (
    <div className="mb-8">
      <div className="flex justify-between items-center mb-6">
        <p className="text-sm font-medium text-gray-700">
          {sectionTitle || `القسم ${currentSection + 1}`}
        </p>
        <p className="text-sm text-gray-600">
          {currentSection + 1} من {totalSections}
        </p>
      </div>

      {/* Step Indicators */}
      <div className="flex justify-between items-center mb-6">
        {Array.from({ length: totalSections }).map((_, index) => {
          // A step is considered completed if:
          // 1. We've already passed it (index < currentSection), OR
          // 2. It's valid and we've reached it (validSections[index] === true AND index <= currentSection)
          const isCompleted = index < currentSection;
          const isCurrent = index === currentSection;

          // Can click if it's a completed step (go back) or current step (stay)
          // Cannot click forward to unvalidated future steps
          const canClick = isCompleted || isCurrent;

          return (
            <div key={index} className="flex items-center flex-1">
              {/* Step Circle */}
              <button
                type="button"
                onClick={() => canClick && onStepClick?.(index)}
                disabled={!canClick}
                className={`relative w-10 h-10 rounded-full font-semibold transition-all ${
                  isCurrent
                    ? "bg-[#e0a735] text-white ring-4 ring-[#e0a735] ring-opacity-30 cursor-default"
                    : isCompleted
                    ? "bg-green-500 text-white cursor-pointer hover:bg-green-600"
                    : "bg-gray-300 text-gray-600 cursor-not-allowed opacity-60"
                }`}
                title={
                  isCurrent
                    ? `القسم الحالي: ${sections?.[index]?.title}`
                    : isCompleted
                    ? `أكتمل: ${sections?.[index]?.title}`
                    : `غير مفعل بعد: ${sections?.[index]?.title}`
                }
              >
                {isCompleted && !isCurrent ? "✓" : index + 1}
              </button>

              {/* Connector Line */}
              {index < totalSections - 1 && (
                <div
                  className={`flex-1 h-1 mx-2 transition-all ${
                    index < currentSection ? "bg-green-500" : "bg-gray-300"
                  }`}
                />
              )}
            </div>
          );
        })}
      </div>

      {/* Progress Bar */}
      <div className="w-full bg-gray-200 rounded-full h-3 overflow-hidden">
        <div
          className="bg-[#e0a735] h-3 rounded-full transition-all duration-500 ease-out"
          style={{ width: `${progress}%` }}
        />
      </div>
    </div>
  );
}
