"use client";

import { ReactNode } from "react";

interface StructuredDataProps {
  data: Record<string, any>;
  id?: string;
}

/**
 * Component to inject structured data (JSON-LD) into the page
 * Should be used in the head of pages that need schema markup
 */
export function StructuredData({
  data,
  id = "structured-data",
}: StructuredDataProps) {
  return (
    <script
      id={id}
      type="application/ld+json"
      dangerouslySetInnerHTML={{
        __html: JSON.stringify(data),
      }}
    />
  );
}

interface SEOHeadProps {
  children?: ReactNode;
  data?: Record<string, any>;
}

/**
 * Component to inject SEO metadata and structured data
 * Use this in page head sections
 */
export function SEOHead({ children, data }: SEOHeadProps) {
  return (
    <>
      {/* Preconnect to important third-party domains */}
      <link rel="preconnect" href="https://fonts.googleapis.com" />
      <link
        rel="preconnect"
        href="https://fonts.gstatic.com"
        crossOrigin="anonymous"
      />

      {/* DNS prefetch for CDN and external services */}
      <link rel="dns-prefetch" href="https://cdn.example.com" />

      {/* Canonical URL */}
      <link
        rel="canonical"
        href={typeof window !== "undefined" ? window.location.href : ""}
      />

      {/* Structured data */}
      {data && <StructuredData data={data} />}

      {/* Additional SEO elements */}
      {children}
    </>
  );
}
