"use client";

import { useState, useEffect } from "react";
import Link from "next/link";
import { usePathname, useRouter } from "next/navigation";
import { getUser, removeAuthToken } from "@/lib/auth";
import { authAPI } from "@/lib/api";
import {
  LayoutDashboard,
  Users,
  FileText,
  Tag,
  Settings,
  LogOut,
  Menu,
  X,
  BarChart3,
  ClipboardList,
} from "lucide-react";

interface SidebarProps {
  role: "admin" | "judge" | "user";
}

export function Sidebar({ role }: SidebarProps) {
  const pathname = usePathname();
  const router = useRouter();
  const [user, setUser] = useState<any>(null);
  const [isOpen, setIsOpen] = useState(false);
  const [mounted, setMounted] = useState(false);

  useEffect(() => {
    setMounted(true);
    setUser(getUser());
  }, []);

  const handleLogout = async () => {
    try {
      await authAPI.logout();
    } catch (error) {
      console.error("Logout error:", error);
    } finally {
      removeAuthToken();
      router.push("/login");
    }
  };

  const adminMenuItems = [
    { href: "/admin", icon: LayoutDashboard, label: "لوحة التحكم" },
    { href: "/admin/users", icon: Users, label: "المستخدمين" },
    { href: "/admin/categories", icon: Tag, label: "التصنيفات" },
    { href: "/admin/languages", icon: FileText, label: "اللغات" },
    { href: "/admin/countries", icon: FileText, label: "الدول" },
    { href: "/admin/governorates", icon: FileText, label: "المحافظات" },
    { href: "/admin/exams", icon: FileText, label: "الاختبارات" },
    {
      href: "/admin/evaluation-sheets",
      icon: ClipboardList,
      label: "نماذج التقييم",
    },
    { href: "/admin/logs", icon: ClipboardList, label: "سجل النظام" },
    { href: "/admin/settings", icon: Settings, label: "الإعدادات" },
  ];

  const judgeMenuItems = [
    { href: "/judge", icon: LayoutDashboard, label: "لوحة التحكم" },
    { href: "/judge/users", icon: Users, label: "المتقدمين" },
    {
      href: "/judge/evaluation-sheets",
      icon: FileText,
      label: "نماذج التقييم",
    },
    { href: "/judge/evaluations", icon: BarChart3, label: "تقييماتي" },
  ];

  const userMenuItems = [
    { href: "/dashboard", icon: LayoutDashboard, label: "لوحة التحكم" },
    { href: "/dashboard/exam", icon: FileText, label: "الاختبار" },
    { href: "/dashboard/profile", icon: Settings, label: "الملف الشخصي" },
  ];

  const menuItems =
    role === "admin"
      ? adminMenuItems
      : role === "judge"
      ? judgeMenuItems
      : userMenuItems;

  // Prevent hydration mismatch
  if (!mounted) {
    return null;
  }

  return (
    <>
      {/* Mobile Menu Button */}
      <button
        onClick={() => setIsOpen(!isOpen)}
        className="lg:hidden fixed top-4 right-4 z-50 p-2 bg-white rounded-lg shadow-lg"
      >
        {isOpen ? <X className="w-6 h-6" /> : <Menu className="w-6 h-6" />}
      </button>

      {/* Overlay */}
      {isOpen && (
        <div
          className="lg:hidden fixed inset-0 bg-black bg-opacity-50 z-40"
          onClick={() => setIsOpen(false)}
        />
      )}

      {/* Sidebar */}
      <aside
        className={`fixed top-0 right-0 h-screen w-64 bg-white shadow-xl z-40 transform transition-transform duration-300 ${
          isOpen ? "translate-x-0" : "translate-x-full"
        } lg:translate-x-0`}
        dir="rtl"
      >
        <div className="flex flex-col h-full">
          {/* Header */}
          <div className="p-6 border-b border-gray-200">
            <h2 className="text-xl font-bold text-gray-900">
              {role === "admin"
                ? "لوحة الإدارة"
                : role === "judge"
                ? "لوحة المقيّم"
                : "لوحة المستخدم"}
            </h2>
            {user && (
              <p className="text-sm text-gray-600 mt-1">{user.full_name}</p>
            )}
          </div>

          {/* Menu Items */}
          <nav className="flex-1 p-4 space-y-2 overflow-y-auto">
            {menuItems.map((item) => {
              const Icon = item.icon;
              const isActive =
                item.href === "/admin"
                  ? pathname === "/admin"
                  : pathname.startsWith(item.href);

              return (
                <Link
                  key={item.href}
                  href={item.href}
                  onClick={() => setIsOpen(false)}
                  className={`flex items-center gap-3 px-4 py-3 rounded-lg transition-colors ${
                    isActive
                      ? "bg-blue-50 text-blue-600 font-medium"
                      : "text-gray-700 hover:bg-gray-50"
                  }`}
                >
                  <Icon className="w-5 h-5" />
                  <span>{item.label}</span>
                </Link>
              );
            })}
          </nav>

          {/* Logout */}
          <div className="p-4 border-t border-gray-200">
            <button
              onClick={handleLogout}
              className="flex items-center gap-3 w-full px-4 py-3 text-red-600 hover:bg-red-50 rounded-lg transition-colors"
            >
              <LogOut className="w-5 h-5" />
              <span>تسجيل الخروج</span>
            </button>
          </div>
        </div>
      </aside>
    </>
  );
}
