/**
 * Generate a browser fingerprint for device identification
 * This creates a unique identifier based on browser/device characteristics
 */
export async function generateBrowserFingerprint(): Promise<string> {
  const fingerprints = [
    // Screen dimensions
    `${window.screen.width}x${window.screen.height}x${window.screen.colorDepth}`,

    // Timezone
    new Date().getTimezoneOffset().toString(),

    // Language
    navigator.language,

    // Platform
    navigator.platform,

    // User agent
    navigator.userAgent,

    // Hardware concurrency (number of CPU cores)
    navigator.hardwareConcurrency?.toString() || "unknown",

    // Device memory
    (navigator as any).deviceMemory?.toString() || "unknown",

    // Max touch points
    navigator.maxTouchPoints?.toString() || "0",

    // Vendor
    navigator.vendor,
  ];

  // Create a hash of all fingerprints
  const fingerprint = fingerprints.join("|");
  const hash = await hashString(fingerprint);

  return hash;
}

/**
 * Simple hash function for fingerprint
 */
async function hashString(str: string): Promise<string> {
  const encoder = new TextEncoder();
  const data = encoder.encode(str);
  const hashBuffer = await crypto.subtle.digest("SHA-256", data);
  const hashArray = Array.from(new Uint8Array(hashBuffer));
  const hashHex = hashArray
    .map((b) => b.toString(16).padStart(2, "0"))
    .join("");
  return hashHex;
}

/**
 * Get device type from user agent
 */
export function getDeviceType(): string {
  const ua = navigator.userAgent;
  if (/mobile|android|iphone|ipad|phone/i.test(ua.toLowerCase())) {
    if (/ipad/i.test(ua)) return "tablet";
    return "mobile";
  }
  return "desktop";
}

/**
 * Get OS from user agent
 */
export function getOS(): string {
  const ua = navigator.userAgent;
  if (/windows/i.test(ua)) return "Windows";
  if (/macintosh|macos/i.test(ua)) return "macOS";
  if (/iphone|ipad|ios/i.test(ua)) return "iOS";
  if (/android/i.test(ua)) return "Android";
  if (/linux/i.test(ua)) return "Linux";
  return "Unknown";
}

/**
 * Get browser name from user agent
 */
export function getBrowser(): string {
  const ua = navigator.userAgent;
  if (/edg/i.test(ua)) return "Edge";
  if (/chrome/i.test(ua)) return "Chrome";
  if (/firefox/i.test(ua)) return "Firefox";
  if (/safari/i.test(ua)) return "Safari";
  if (/opera|opr/i.test(ua)) return "Opera";
  return "Unknown";
}
